/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test change delay for SystemC-AMS 2.0 dtdf (two inport delays + initialize
// with second module rate 7 and outport delay in mod_a

#include <systemc-ams>
#include "test_utilities.h"

void check_trace_file(std::string name)
{
  std::fstream fin;

  fin.open(name.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open trace file: " << name;
    SC_REPORT_ERROR("test", str.str().c_str());
    return;
  }

  std::string header;
  std::getline(fin, header);

  long cnt = 0;

  while (!fin.eof())
  {
    std::string line;

    std::getline(fin, line);
    std::istringstream line_str(line);

    double time;
    long   value;

    line_str >> time;
    if (line_str.fail()) continue; // ignore empty lines

    line_str >> value;

    if (value != cnt)
    {
      std::ostringstream str;
      str << "Unexpexted value in trace file at time: " << time;
      str << " expected value: " << cnt << " read value: " << value;
      SC_REPORT_ERROR("test", str.str().c_str());
      return;
    }

    cnt++;
  }
}

bool is_time_not_equal(const sca_core::sca_time& t1, const sca_core::sca_time& t2)
{
  sca_core::sca_time tdiff;
  if (t1 > t2) tdiff = t1 - t2;
  else         tdiff = t2 - t1;
  return tdiff > sc_core::sc_get_time_resolution();
}

SCA_TDF_MODULE( mod_a )
{
  sca_tdf::sca_out<long> outp;

  void print_status()
  {
    (*logstream) << "   are_attribute_changes_allowed: " << (are_attribute_changes_allowed() ? "yes" : "no") << std::endl;
    (*logstream) << "   is_dynamic:                    " << (is_dynamic() ? "yes" : "no") << std::endl;
    (*logstream) << "   are_attributes_changed:        " << (are_attributes_changed() ? "yes" : "no") << std::endl;
    (*logstream) << "   is_timestep_changed:           " << (is_timestep_changed() ? "yes" : "no") << std::endl;
    (*logstream) << "   get_timestep:                  " << get_timestep() << std::endl;
    (*logstream) << "   get_last_timestep:             " << get_last_timestep() << std::endl;
    (*logstream) << "   get_max_timestep:              ";
    if (get_max_timestep() == sca_core::sca_max_time()) (*logstream) << "sca_core::sca_max_time" << std::endl;
    else                                               (*logstream) << get_max_timestep() << std::endl;

    (*logstream) << "   get_time:                      " << get_time() << std::endl;
    (*logstream) << std::endl;
    (*logstream) << "   outp.get_delay()               " << outp.get_delay() << std::endl;
    (*logstream) << "   outp.get_rate()                " << outp.get_rate() << std::endl;
    (*logstream) << "   outp.get_time()                " << outp.get_time() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      (*logstream) << "   outp.get_time(" << i << ")               " << outp.get_time(i) << std::endl;
    }

    (*logstream) << "   outp.get_timestep()            " << outp.get_timestep() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      (*logstream) << "   outp.get_timestep(" << i << ")           " << outp.get_timestep(i) << std::endl;
    }

    (*logstream) << "   outp.get_last_timestep()       " << outp.get_last_timestep() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      (*logstream) << "   outp.get_last_timestep(" << i << ")      " << outp.get_last_timestep(i) << std::endl;
    }

    (*logstream) << "   outp.get_max_timestep()        ";
    if (outp.get_max_timestep() == sca_core::sca_max_time()) (*logstream) << "sca_core::sca_max_time" << std::endl;
    else                                               (*logstream) << outp.get_max_timestep() << std::endl;

    (*logstream) << "   outp.is_timestep_changed()     " << (outp.is_timestep_changed() ? "yes" : "no") << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      (*logstream) << "   outp.is_timestep_changed(" << i << ")    " << (outp.is_timestep_changed(i) ? "yes" : "no") << std::endl;
    }

    (*logstream) << "   outp.is_rate_changed()         " << (outp.is_rate_changed() ? "yes" : "no") << std::endl;
    (*logstream) << "   outp.is_delay_changed()        " << (outp.is_delay_changed() ? "yes" : "no") << std::endl;
  }

  void set_attributes()
  {
    outp.set_rate(3);
    outp.set_delay(4);

    (*logstream) << "set_attributes() mod_a" << std::endl;

    set_timestep(1.0, sc_core::SC_MS);
    accept_attribute_changes();

    (*logstream) << "      outp.set_rate(3);" << std::endl;
    (*logstream) << "      set_timestep(1.0,sc_core::SC_MS);" << std::endl;
    (*logstream) << "      accept_attribute_changes();\n" << std::endl;
  }

  void initialize()
  {
    (*logstream) << "--------------------------------------------" << std::endl;
    (*logstream) << get_time() << " initialize() mod_a" << std::endl;
    print_status();
    out_cnt = 0;

    last_time_step = this->get_timestep();
    last_time = this->get_time() - this->get_timestep(); // for the first timestep the propagated value is returned
    last_port_time = outp.get_time() - outp.get_timestep();
    last_port_timestep = outp.get_timestep();

    for (unsigned long i = 0; i < outp.get_delay(); i++)
    {
      outp.initialize(out_cnt, i);
      out_cnt++;
    }
  }

  void reinitialize()
  {
    (*logstream) << "--------------------------------------------" << std::endl;
    (*logstream) << "--------------------------------------------" << std::endl;
    (*logstream) << get_time() << " reinitialize() mod_a" << std::endl;
    print_status();

    out_cnt -= outp.get_delay();
    for (unsigned long i = 0; i < outp.get_delay(); i++)
    {
      outp.initialize(out_cnt, i);
      out_cnt++;
    }
  }

  void processing()
  {
    (*logstream) << "--------------------------------------------" << std::endl;
    (*logstream) << get_time() << " processing() mod_a" << std::endl;
    print_status();

    for (unsigned long i = 0; i < outp.get_rate(); ++i)
    {
      outp.write(long(out_cnt), i);
      out_cnt++;
    }

    //// consistency checks ///////////////////////////

    for (unsigned long i = 0; i < outp.get_rate(); ++i)
    {
      if (is_time_not_equal(outp.get_time(i) - last_port_time, outp.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep expect: " << outp.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << outp.get_time(i);
        str << " get_timestep: " << outp.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_time = outp.get_time(i);

      if (is_time_not_equal(last_port_timestep, outp.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep expect: " << last_port_timestep;
        str << " outp.get_last_timestep: " << outp.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_timestep = outp.get_timestep(i);
    }

    if (is_time_not_equal(this->get_time() - last_time, this->get_timestep()))
    {
      std::ostringstream str;
      str << "Wrong timestep expect: " << this->get_time() - last_time;
      str << " last_time: " << last_time << " current time: " << this->get_time();
      str << " get_timestep: " << this->get_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time = this->get_time();

    if (is_time_not_equal(last_time_step, this->get_last_timestep()))
    {
      std::ostringstream str;
      str << "Wrong last timestep expect: " << last_time_step;
      str << " get_last_timestep: " << this->get_last_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time_step = this->get_timestep();
  }

  void change_attributes()
  {

    (*logstream) << std::endl << std::endl;
    (*logstream)  << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl;
    (*logstream)  << std::endl;
    (*logstream)  << "--------------------------------------------" << std::endl;
    (*logstream)  << get_time() << " change_attributes() mod_a  cnt: " << cnt << std::endl;

    switch(cnt)
    {
      case 12:
        out_cnt += 0 - outp.get_delay();
        outp.set_delay(0);
        break;
      case 15:
        out_cnt += 3 - outp.get_delay();
        outp.set_delay(3);
        break;
      case 21:
        out_cnt += 0 - outp.get_delay();
        outp.set_delay(0);
        break;
      case 26:
        out_cnt += 2 - outp.get_delay();
        outp.set_delay(2);
        break;
      case 29:
        out_cnt += 9 - outp.get_delay();
        outp.set_delay(9);
        break;

      default:
        break;
    }

    cnt++;
    print_status();
  }

  SCA_CTOR( mod_a ) 
  : outp("outp"), cnt(0), out_cnt(0), logstream(&std::cout)
  {}

  std::ostream* logstream;

 private:
  long cnt;
  long out_cnt;

  sca_core::sca_time last_time;
  sca_core::sca_time last_time_step;

  sca_core::sca_time last_port_time;
  sca_core::sca_time last_port_timestep;
};

////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE( mod_b )
{
  sca_tdf::sca_in<long> inp;

  void print_status()
  {
    (*logstream) << std::setw(40)<< "" << "   are_attribute_changes_allowed: " << (are_attribute_changes_allowed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(40)<< "" << "   is_dynamic:                    " << (is_dynamic() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(40)<< "" << "   are_attributes_changed:        " << (are_attributes_changed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(40)<< "" << "   is_timestep_changed:           " << (is_timestep_changed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(40)<< "" << "   get_timestep:                  " << get_timestep() << std::endl;
    (*logstream) << std::setw(40)<< "" << "   get_last_timestep:             " << get_last_timestep() << std::endl;
    (*logstream) << std::setw(40)<< "" << "   get_max_timestep:              ";
    if (get_max_timestep() == sca_core::sca_max_time()) (*logstream) << "sca_core::sca_max_time" << std::endl;
    else                                               (*logstream) << get_max_timestep() << std::endl;

    (*logstream) << std::setw(40)<< "" << "   get_time:                      " << get_time() << std::endl;
    (*logstream) << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_delay()                " << inp.get_delay() << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_rate()                 " << inp.get_rate() << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_time()                 " << inp.get_time() << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_time(1)                " << inp.get_time(1) << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_timestep()             " << inp.get_timestep() << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_timestep(1)            " << inp.get_timestep(1) << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_last_timestep()        " << inp.get_last_timestep() << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_last_timestep(1)       " << inp.get_last_timestep(1) << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.get_max_timestep()         ";
    if (inp.get_max_timestep() == sca_core::sca_max_time()) (*logstream) << "sca_core::sca_max_time" << std::endl;
    else                                               (*logstream) << inp.get_max_timestep() << std::endl;

    (*logstream) << std::setw(40)<< "" << "   inp.is_timestep_changed()      " << (inp.is_timestep_changed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.is_timestep_changed(1)     " << (inp.is_timestep_changed(1) ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.is_rate_changed()          " << (inp.is_rate_changed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(40)<< "" << "   inp.is_delay_changed()         " << (inp.is_delay_changed() ? "yes" : "no") << std::endl;
  }

  void set_attributes()
  {
    inp.set_rate(2);
    inp.set_delay(3);

    (*logstream) << std::setw(40) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(40) << "" << " set_attributes() mod_b" << std::endl;
    (*logstream) << std::setw(40) << "" << "     inp.set_rate(2);" << std::endl;
  }

  void initialize()
  {
    (*logstream) << std::setw(40) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(40) << "" << get_time() << " initialize() mod_b" << std::endl;
    print_status();

    last_time_step = this->get_timestep();
    last_time = this->get_time() - this->get_timestep(); // for the first timestep the propagated value is returned
    last_port_time = inp.get_time() - inp.get_timestep();
    last_port_timestep = inp.get_timestep();

    in_cnt = -inp.get_delay();
    for (unsigned long i = 0; i < inp.get_delay(); ++i)
    {
      inp.initialize(in_cnt, i);
      in_cnt++;
    }
    in_cnt = -inp.get_delay();
  }

  void reinitialize()
  {
    (*logstream) << std::setw(40) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(40) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(40) << "" << get_time() << " reinitialize() mod_b" << std::endl;
    print_status();

    for (unsigned long i = 0; i < inp.get_delay(); ++i)
    {
      inp.initialize(cnt+i, i);
    }
    initialize_cnt = 0;
  }

  void processing()
  {
    (*logstream) << std::setw(40) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(40) << "" << get_time() << " processing() mod_b" << std::endl;
    print_status();

    for (unsigned long i = 0; i < inp.get_rate(); ++i, ignore--)
    {
      long expected_value = in_cnt;

      if (initialize_cnt < inp.get_delay())
      {
        expected_value = cnt+initialize_cnt;
        initialize_cnt++;
      }
      else
      {
        initialize_cnt = 9999999;
      }

      if (expected_value != long(inp.read(i)))
      {
        std::ostringstream str;
        str << " read wrong value for sample: " << i << " value: " << inp.read(i)
            << " expected " << in_cnt << " cnt: " << cnt
            << " module: " << name() << " initialize_cnt: " << initialize_cnt-1 << std::endl;
        SC_REPORT_ERROR("test", str.str().c_str());
      }
      in_cnt++;
    }

    for (unsigned long i = 0; i < inp.get_rate(); ++i)
    {
      if (is_time_not_equal(inp.get_time(i) - last_port_time, inp.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep expect: " << inp.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << inp.get_time(i);
        str << " get_timestep: " << inp.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_time = inp.get_time(i);

      if (is_time_not_equal(last_port_timestep, inp.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep expect: " << last_port_timestep;
        str << " outp.get_last_timestep: " << inp.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_timestep = inp.get_timestep(i);
    }

    if (is_time_not_equal(this->get_time() - last_time,  this->get_timestep()))
    {
      std::ostringstream str;
      str << "Wrong timestep expect: " << this->get_time() - last_time;
      str << " last_time: " << last_time << " current time: " << this->get_time();
      str << " get_timestep: " << this->get_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time = this->get_time();

    // consistency check
    if (is_time_not_equal(last_time_step, this->get_last_timestep()))
    {
      std::ostringstream str;
      str << "Wrong last timestep expect: " << last_time_step;
      str << " get_last_timestep: " << this->get_last_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time_step = this->get_timestep();
  }

  void change_attributes()
  {
    accept_attribute_changes();
    cnt++;
    (*logstream)<< std::setw(40)  << ""  << std::endl;
    (*logstream)<< std::setw(40)  << ""  << "--------------------------------------------" << std::endl;
    (*logstream)<< std::setw(40)  << ""  << get_time() << " change_attributes() mod_b  cnt: " << cnt << std::endl;

    switch(cnt)
    {
      case 1:
        break;
      case 2:
        // the order has no influence, due consistency check is after
        // change_attribute execution
        does_attribute_changes();
        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    does_attribute_changes();" << std::endl;
        break;
      case 3:
        does_no_attribute_changes();
        inp.set_delay(1);  // from 3 -> 1
        in_cnt += 2; // two delay values removed
        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    does_no_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    outp.set_delay(1);" << std::endl;
        break;
      case 4:
        does_attribute_changes();
        (*logstream)<< std::setw(40) << ""  << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40) << ""  << "    does_attribute_changes();" << std::endl;
        break;
      case 5:
        request_next_activation(0.5, sc_core::SC_MS);
        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    request_next_activation(0.5,sc_core::SC_MS);" << std::endl;
        break;
      case 6:
        does_attribute_changes();
        inp.set_delay(5);  // from 1->5
        in_cnt -= 4; // 5 sample delay added
        ignore = 4;
        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream) << std::setw(40)  << "" << "    inp.set_delay(5);"         << std::endl;
        break;
      case 7:
      case 8:
      case 9:
      case 10:
      case 11: break;
      case 12:
        does_attribute_changes();
        inp.set_delay(2);  // from 5->2
        in_cnt += 3; // remove 3 sample delay
        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    inp.set_delay(2);" << std::endl;
        break;
      case 13:
        does_attribute_changes();
        inp.set_delay(3);  // from 2->3
        in_cnt -= 1; // add one sample delay
        ignore = 1;
        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    inp.set_delay(3);" << std::endl;
        break;
      case 14:
      case 15: break;
      case 16:
        does_attribute_changes();
        inp.set_delay(0);  // from 3->0
        in_cnt += 3; // remove 3 sample delay

        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    inp.set_delay(0);" << std::endl;
        break;
      case 17:
        does_attribute_changes();
        inp.set_delay(3);  // from 0->3
        in_cnt -= 3; // add one sample delay
        ignore = 3;
        (*logstream)<< std::setw(40)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(40)  << "" << "    inp.set_delay(3);" << std::endl;
        break;
      case 18:
      case 19:
      case 20:
      case 21:
      case 22: break;

      default:
        break;
    }

    print_status();
  }

  SCA_CTOR( mod_b ) 
  : logstream(&std::cout)
  {
    cnt = 0;
    ignore = 0;
    initialize_cnt = 9999999;
    in_cnt = 0;
  }

  long cnt;
  long in_cnt;
  long ignore;
  long initialize_cnt;

  sca_core::sca_time last_time;
  sca_core::sca_time last_time_step;

  sca_core::sca_time last_port_time;
  sca_core::sca_time last_port_timestep;

  std::ostream* logstream;
};

////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE( mod_c )
{
  sca_tdf::sca_in<long> inp;

  void print_status()
  {
    (*logstream) << std::setw(60)<< "" << "   are_attribute_changes_allowed: " << (are_attribute_changes_allowed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(60)<< "" << "   is_dynamic:                    " << (is_dynamic() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(60)<< "" << "   are_attributes_changed:        " << (are_attributes_changed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(60)<< "" << "   is_timestep_changed:           " << (is_timestep_changed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(60)<< "" << "   get_timestep:                  " << get_timestep() << std::endl;
    (*logstream) << std::setw(60)<< "" << "   get_last_timestep:             " << get_last_timestep() << std::endl;
    (*logstream) << std::setw(60)<< "" << "   get_max_timestep:              ";
    if (get_max_timestep() == sca_core::sca_max_time()) (*logstream) << "sca_core::sca_max_time" << std::endl;
    else                                               (*logstream) << get_max_timestep() << std::endl;

    (*logstream) << std::setw(60)<< "" << "   get_time:                      " << get_time() << std::endl;
    (*logstream) << std::endl;
    (*logstream) << std::setw(60)<< "" << "   inp.get_delay()                " << inp.get_delay() << std::endl;
    (*logstream) << std::setw(60)<< "" << "   inp.get_rate()                 " << inp.get_rate() << std::endl;
    (*logstream) << std::setw(60)<< "" << "   inp.get_time()                 " << inp.get_time() << std::endl;
    for (unsigned long i = 0; i < inp.get_rate(); i++)
    {
      (*logstream) << std::setw(60)<< "" << "   inp.get_time(" << i << ")           " << inp.get_time(i) << std::endl;
    }

    (*logstream) << std::setw(60)<< "" << "   inp.get_timestep()             " << inp.get_timestep() << std::endl;
    for (unsigned long i = 0; i < inp.get_rate(); i++)
    {
      (*logstream) << std::setw(60)<< "" << "   inp.get_timestep(" << i << ")           " << inp.get_timestep(i) << std::endl;
    }

    (*logstream) << std::setw(60)<< "" << "   inp.get_last_timestep()        " << inp.get_last_timestep() << std::endl;
    for (unsigned long i = 0; i < inp.get_rate(); i++)
    {
      (*logstream) << std::setw(60)<< "" << "   inp.get_last_timestep(" << i << ")           " << inp.get_last_timestep(i) << std::endl;
    }

    (*logstream) << std::setw(60)<< "" << "   inp.get_max_timestep()         ";
    if (inp.get_max_timestep() == sca_core::sca_max_time()) (*logstream) << "sca_core::sca_max_time" << std::endl;
    else                                               (*logstream) << inp.get_max_timestep() << std::endl;

    (*logstream) << std::setw(60)<< "" << "   inp.is_timestep_changed()      " << (inp.is_timestep_changed() ? "yes" : "no") << std::endl;
    for (unsigned long i = 0; i < inp.get_rate(); i++)
    {
      (*logstream) << std::setw(60)<< "" << "   inp.is_timestep_changed(" << i << ")      " << (inp.is_timestep_changed(i) ? "yes" : "no") << std::endl;
    }

    (*logstream) << std::setw(60)<< "" << "   inp.is_rate_changed()          " << (inp.is_rate_changed() ? "yes" : "no") << std::endl;
    (*logstream) << std::setw(60)<< "" << "   inp.is_delay_changed()         " << (inp.is_delay_changed() ? "yes" : "no") << std::endl;
  }

  void set_attributes()
  {
    inp.set_rate(7);
    inp.set_delay(3);

    (*logstream) << std::setw(60) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(60) << "" << " set_attributes() mod_c" << std::endl;
    (*logstream) << std::setw(60) << "" << "     inp.set_rate(2);" << std::endl;
  }

  void initialize()
  {
    (*logstream) << std::setw(60) << "" <<"--------------------------------------------" << std::endl;
    (*logstream) << std::setw(60) << "" << get_time() << " initialize() mod_c" << std::endl;
    print_status();

    last_time_step = this->get_timestep();
    last_time = this->get_time() - this->get_timestep(); // for the first timestep the propagated value is returned
    last_port_time = inp.get_time() - inp.get_timestep();
    last_port_timestep = inp.get_timestep();

    in_cnt = -inp.get_delay();
    for (unsigned long i = 0; i< inp.get_delay(); ++i)
    {
      inp.initialize(in_cnt, i);
      in_cnt++;
    }
    in_cnt = -inp.get_delay();
  }

  void reinitialize()
  {
    (*logstream) << std::setw(60) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(60) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(60) << "" << get_time() << " reinitialize() mod_c" << std::endl;
    print_status();

    switch(cnt)
    {
      case 6:
      case 13:
      case 17:
      case 26:
      case 33:
      case 37:
      {
        for (unsigned long i = 0; i < inp.get_delay(); ++i)
        {
          inp.initialize(cnt + i + 1, i);
        }
        initialize_cnt = 0;
      }
        break;
      default:
        break;
    }
  }

  void processing()
  {
    (*logstream) << std::setw(60) << "" << "--------------------------------------------" << std::endl;
    (*logstream) << std::setw(60) << "" << get_time() << " processing() mod_c" << std::endl;
    print_status();

    for (unsigned long i = 0; i < inp.get_rate(); ++i, ignore--)
    {
      long expected_value = in_cnt;

      if (initialize_cnt < inp.get_delay())
      {
        expected_value = cnt + initialize_cnt + 1;
        initialize_cnt++;
      }
      else
      {
        initialize_cnt = 9999999;
      }

      if (expected_value != long(inp.read(i)))
      {
        std::ostringstream str;
        str << " read wrong value for sample: " << i << " value: " << inp.read(i)
            << " expected " << in_cnt << " cnt: " << cnt
            << " module: " << name() << " initialize_cnt: " << initialize_cnt-1 << std::endl;
        SC_REPORT_ERROR("test", str.str().c_str());
      }
      in_cnt++;
    }

    for (unsigned long i = 0; i < inp.get_rate(); ++i)
    {
      if (is_time_not_equal(inp.get_time(i) - last_port_time, inp.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep expect: " << inp.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << inp.get_time(i);
        str << " get_timestep: " << inp.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_time = inp.get_time(i);

      if (is_time_not_equal(last_port_timestep, inp.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep expect: " << last_port_timestep;
        str << " inp.get_last_timestep: " << inp.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_timestep = inp.get_timestep(i);
    }

    if (is_time_not_equal(this->get_time() - last_time,  this->get_timestep()))
    {
      std::ostringstream str;
      str << "Wrong timestep expect: " << this->get_time() - last_time;
      str << " last_time: " << last_time << " current time: " << this->get_time();
      str << " get_timestep: " << this->get_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());

    }
    last_time = this->get_time();

    // consistency check
    if (is_time_not_equal(last_time_step, this->get_last_timestep()))
    {
      std::ostringstream str;
      str << "Wrong last timestep expect: " << last_time_step;
      str << " get_last_timestep: " << this->get_last_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time_step = this->get_timestep();
  }

  void change_attributes()
  {
    accept_attribute_changes();
    cnt++;
    (*logstream)<< std::setw(60)  << ""  << std::endl;
    (*logstream)<< std::setw(60)  << ""  << "--------------------------------------------" << std::endl;
    (*logstream)<< std::setw(60)  << ""  << get_time() << " change_attributes() mod_c  cnt: " << cnt << std::endl;

    switch(cnt)
    {
      case 1:
        break;
      case 2:
        // the order has no influence, due consistency check is after
        // change_attribute execution
        does_attribute_changes();
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        break;
      case 3:
        does_no_attribute_changes();
        inp.set_delay(1);  // from 3 -> 1
        in_cnt += 2; // two delay values removed
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_no_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    outp.set_delay(1);" << std::endl;
        break;
      case 4:
        does_attribute_changes();
        (*logstream)<< std::setw(60) << ""  << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60) << ""  << "    does_attribute_changes();" << std::endl;
        break;
      case 5:
        request_next_activation(0.5, sc_core::SC_MS);
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    request_next_activation(0.5,sc_core::SC_MS);" << std::endl;
        break;
      case 6:
        does_attribute_changes();
        inp.set_delay(5);  // from 1->5
        in_cnt -= 4; // 5 sample delay added
        ignore = 4;
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream) << std::setw(60)  << "" << "    inp.set_delay(5);"         << std::endl;
        break;
      case 7:
      case 8:
      case 9:
      case 10:
      case 11: break;
      case 12:
        does_attribute_changes();
        inp.set_delay(2);  // from 5->2
        in_cnt += 3; // remove 3 sample delay
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(2);" << std::endl;
        break;
      case 13:
        does_attribute_changes();
        inp.set_delay(3);  // from 2->3
        in_cnt -= 1; // add one sample delay
        ignore = 1;
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(3);" << std::endl;
        break;
      case 14:
      case 15: break;
      case 16:
        does_attribute_changes();
        inp.set_delay(0);  // from 3->0
        in_cnt += 3; // remove 3 sample delay
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(0);" << std::endl;
        break;
      case 17:
        does_attribute_changes();
        inp.set_delay(3);  // from 0->3
        in_cnt -= 3; // add one sample delay
        ignore = 3;
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(3);" << std::endl;
        break;
      case 18:
      case 19:
      case 20:
      case 21: break;

      /////////////////////////////////////////////////////////////////////////////

      case 22:
        // the order has no influence, due consistency check is after
        // change_attribute execution
        does_attribute_changes();
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        break;
      case 23:
        does_no_attribute_changes();
        inp.set_delay(2);  // from 3 -> 2
        in_cnt += 1; // two delay values removed
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_no_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    outp.set_delay(2);" << std::endl;
        break;
      case 24:
        does_attribute_changes();
        (*logstream)<< std::setw(60) << ""  << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60) << ""  << "    does_attribute_changes();" << std::endl;
        break;
      case 25:
        request_next_activation(0.5, sc_core::SC_MS);
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    request_next_activation(0.5,sc_core::SC_MS);" << std::endl;
        break;
      case 26:
        does_attribute_changes();
        inp.set_delay(4);  // from 2->4
        in_cnt -= 2; // 5 sample delay added
        ignore = 2;
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream) << std::setw(60)  << "" << "    inp.set_delay(4);"         << std::endl;
        break;
      case 27:
      case 28:
      case 29:
      case 30:
      case 31: break;
      case 32:
        does_attribute_changes();
        inp.set_delay(2);  // from 4->2
        in_cnt += 2; // remove 3 sample delay
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(2);" << std::endl;
        break;
      case 33:
        does_attribute_changes();
        inp.set_delay(3);  // from 2->3
        in_cnt -= 1; // add one sample delay
        ignore = 1;
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(3);" << std::endl;
        break;
      case 34:
      case 35: break;
      case 36:
        does_attribute_changes();
        inp.set_delay(0);  // from 3->0
        in_cnt += 3; // remove 3 sample delay
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(0);" << std::endl;
        break;
      case 37:
        does_attribute_changes();
        inp.set_delay(3);  // from 0->3
        in_cnt -= 3; // add one sample delay
        ignore = 3;
        (*logstream)<< std::setw(60)  << "" << "--------------------------------------------" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    does_attribute_changes();" << std::endl;
        (*logstream)<< std::setw(60)  << "" << "    inp.set_delay(3);" << std::endl;
        break;

      default:
        sc_core::sc_pause();
        return;
    }

    print_status();
  }

  SCA_CTOR( mod_c ) 
  : inp("inp"), cnt(0), ignore(0),
    initialize_cnt(9999999), in_cnt(0), logstream(&std::cout)
  {}

  std::ostream* logstream;

 private:
  long cnt;
  long in_cnt;
  long ignore;
  long initialize_cnt;

  sca_core::sca_time last_time;
  sca_core::sca_time last_time_step;

  sca_core::sca_time last_port_time;
  sca_core::sca_time last_port_timestep;
};

/////////////////////////////////////////////////////////////////////////////

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  std::cout << "| mod_a outp|3--->2|inp  mod_b |" << std::endl;
  std::cout << "               |->7|inp  mod_c |" << std::endl;
  std::cout << std::endl;

  sca_tdf::sca_signal<long> s1;

  mod_a a("a");
  a.outp(s1);

  mod_b b("b");
  b.inp(s1);

  mod_c c("c");
  c.inp(s1);

  // redirect logs of modules to different streams
  std::ostringstream sa, sb, sc;
  a.logstream = &sa;
  b.logstream = &sb;
  c.logstream = &sc;

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test08.dat");
  sca_util::sca_trace(tf, s1, "s1");

  sc_core::sc_start();

  sca_util::sca_close_tabular_trace_file(tf);

  check_trace_file("test08.dat");

  // print out logs one module after the other - to become
  // scheduling order independent
  std::cout << sa.str();
  std::cout << sb.str();
  std::cout << sc.str();

  TEST_LABEL_END;

  sc_core::sc_stop();


  return 0;
}
