/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> out; // output port

  sin_src( sc_core::sc_module_name nm, double ampl_= 1.0, double freq_ = 1.0e3,
           sca_core::sca_time Tm_ = sca_core::sca_time(0.125, sc_core::SC_MS) )
  : out("out"), ampl(ampl_), freq(freq_), Tm(Tm_)
  {}

  void set_attributes()
  {
    out.set_timestep(Tm);
  }

  void processing()
  {
    double t = get_time().to_seconds(); // actual time
    out.write(ampl * std::sin( 2.0 * M_PI * freq * t ) ) ;
  }

 private:
  double ampl; // amplitude
  double freq; // frequency
  sca_core::sca_time Tm; // module time step
};

SC_MODULE(lsf_in_tdf)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  sca_lsf::sca_add add1;
  sca_lsf::sca_dot dot1;
  sca_lsf::sca_gain gain1;
  sca_lsf::sca_tdf::sca_source tdf2lsf;
  sca_lsf::sca_tdf::sca_sink lsf2tdf;

  lsf_in_tdf( sc_core::sc_module_name, double k, double k2 )
  : in("in"), out("out"), add1("add1"), dot1("dot1", k), gain1("gain1", k2), 
    tdf2lsf("tdf2lsf"), lsf2tdf("lsf2tdf"),
    sig1("sig1"), sig2("sig2"), sig3("sig3"), sig4("sig4")
  {
    tdf2lsf.inp(in);
    tdf2lsf.y(sig1);

    add1.x1(sig1);
    add1.x2(sig3);
    add1.y(sig2);

    dot1.x(sig2);
    dot1.y(sig4);

    gain1.x(sig4);
    gain1.y(sig3);

    lsf2tdf.x(sig4);
    lsf2tdf.outp(out);
  }
 private:
  sca_lsf::sca_signal sig1, sig2, sig3, sig4;
};


int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;
  
  sin_src src("src");
  src.out(sig1);
  
  lsf_in_tdf enc_lsf("enc_lsf", 1.0, 1.0); 
    enc_lsf.in(sig1);
    enc_lsf.out(sig2);
    
  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("lsf_in_tdf");

  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");
  
  sc_core::sc_start(5, sc_core::SC_MS);
  
  sca_util::sca_close_tabular_trace_file(tf);

  test_util::check_results("lsf_in_tdf", 2, 1e-14);

  TEST_LABEL_END;
  
  return 0;  
}

