/**
 *
 * @file tlm_mesh_2d.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: tlm_mesh_2d.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef ASEBT_TLM_MESH_2_D_HH
#define ASEBT_TLM_MESH_2_D_HH

#include "sc_tlm_1/tlm_mesh_router.hh"

#include "tlm.h"
#ifdef MTI_SYSTEMC
#include "simple_initiator_socket.h"
#include "simple_target_socket.h"
#else
#include "tlm_utils/simple_initiator_socket.h"
#include "tlm_utils/simple_target_socket.h"
#endif
#include <systemc>

#include <iostream>

namespace asebt
{
   namespace mesh_2d_sc_tlm_1
   {

      template<unsigned int data_width_g = 32,
	       unsigned int rows_g = 2,
	       unsigned int cols_g = 2>
      class TlmMesh2D : public sc_core::sc_module
      {
      public:

	 SC_HAS_PROCESS(TlmMesh2D);	 

	 // Sockets: N, W, S, E, IP
	 tlm_utils::simple_initiator_socket_tagged
	 <TlmMesh2D, data_width_g> initSockets[rows_g * cols_g];
	 tlm_utils::simple_target_socket_tagged
	 <TlmMesh2D, data_width_g> targetSockets[rows_g * cols_g];

	 //* Constructor
	 TlmMesh2D(sc_core::sc_module_name name)
	    : sc_core::sc_module(name),
	      _cycleTime(20, sc_core::SC_NS)
	 {
	    for(unsigned int r = 0; r < rows_g; ++r)
	    {
	       for(unsigned int c = 0; c < cols_g; ++c)
	       {
		  std::ostringstream oss;
		  oss << "router_r" << r << "_c" << c;
		  unsigned int i = r * cols_g + c;
		  _routers[i] = 
		     new TlmMeshRouter<data_width_g>
		     (oss.str().c_str(), r, c, rows_g, cols_g);

		  std::ostringstream oss2;
		  oss2 << "router_r" << r << "_c" << c << "_ip_peq";
		  _ip_peq[i] = new tlm_utils::peq_with_get
		     <tlm::tlm_generic_payload>(oss2.str().c_str());

		  oss2.str("");
		  oss2 << "router_r" << r << "_c" << c << "_noc_peq";
		  _noc_peq[i] = new tlm_utils::peq_with_get
		     <tlm::tlm_generic_payload>(oss2.str().c_str());
		  
		  initSockets[i].register_nb_transport_bw
		     (this, &TlmMesh2D::ip_side_bw, i);
		  targetSockets[i].register_nb_transport_fw
		     (this, &TlmMesh2D::ip_side_fw, i);
		  sc_spawn(sc_bind(&TlmMesh2D::ip_side_thread, this, i));

		  _init[i].register_nb_transport_bw
		     (this, &TlmMesh2D::noc_side_bw, i);
		  _target[i].register_nb_transport_fw
		     (this, &TlmMesh2D::noc_side_fw, i);
		  sc_spawn(sc_bind(&TlmMesh2D::noc_side_thread, this, i));

		  _init[i].bind(*(_routers[i]->targetSockets[4]));
		  (*(_routers[i]->initSockets[4])).bind(_target[i]);

		  if(c > 0)
		  {
		     unsigned int t = r * cols_g + c - 1;
		     (*_routers[i]->initSockets[1]).
			bind(*_routers[t]->targetSockets[3]);
		     (*_routers[t]->initSockets[3]).
			bind(*_routers[i]->targetSockets[1]);
		  }
		  if(r > 0)
		  {
		     unsigned int t = (r-1) * cols_g + c;
		     (*_routers[i]->initSockets[0]).
			bind(*_routers[t]->targetSockets[2]);
		     (*_routers[t]->initSockets[2]).
			bind(*_routers[i]->targetSockets[0]);
		  }
	       }
	    }	 	 
	 }
	

	 //* Destructor
	 ~TlmMesh2D()
	 {
	    for(unsigned int i = 0; i < rows_g * cols_g; ++i)
	    {
	       delete _routers[i]; _routers[i] = 0;
	       delete _noc_peq[i]; _noc_peq[i] = 0;
	       delete _ip_peq[i];  _ip_peq[i]  = 0;
	    }
	 }

      private:


	 void ip_side_thread(unsigned int agent)
	 {
	    tlm::tlm_generic_payload* trans = 0;
	    tlm::tlm_phase            phase;
	    sc_core::sc_time          delay;
	    tlm::tlm_sync_enum        retval;
	    unsigned int              source = 0;

	    //std::cout << "TlmMesh2D::ip_thread(" << agent << ")" << std::endl;

	    // std::ostringstream oss;
	    // oss << "at_" << agent << "_from_ip.txt";
	    // std::ofstream ofs(oss.str().c_str());

	    while(true)
	    {
	       // Check for pending transactions
	       if((trans = _ip_peq[agent]->get_next_transaction()) == 0)
	       { 		  
		  wait(_ip_peq[agent]->get_event());
		  trans = _ip_peq[agent]->get_next_transaction();
	       }
	       
	       // ofs << "Agent " << agent << " sending data  "
	       // 	   << sc_core::sc_time_stamp().value() << std::endl;

	       phase = tlm::END_REQ;
	       delay = _cycleTime * 
		  ((trans->get_data_length() / trans->get_streaming_width()) + 
		     2 );

	       retval = targetSockets[agent]->nb_transport_bw(*trans, 
							       phase, delay);
	    
	       if(retval != tlm::TLM_COMPLETED)
	       {
		  std::ostringstream oss;
		  oss << "TlmMesh2Dr::ip_thread : Not supporting responses";
		  throw std::runtime_error(oss.str().c_str());
	       }
	    
	       // Forward transaction

	       phase = tlm::BEGIN_REQ;
	       delay = _cycleTime * 7;

	       //wait(delay);
	    
	       retval = _init[agent]->nb_transport_fw(*trans, phase, delay);
	    
	       if(retval == tlm::TLM_ACCEPTED || retval == tlm::TLM_UPDATED)
	       {
		  if(phase == tlm::BEGIN_REQ)
		  {	
		     wait(_noctxCompleteEvent[agent]);		
		  }
		  else if(phase == tlm::END_REQ)
		  {
		     std::ostringstream oss;
		     oss << "TlmMesh2D::thread : END_REQ not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else if(phase == tlm::BEGIN_RESP)
		  {
		     std::ostringstream oss;
		     oss << "TlmMesh2D::thread : BEGIN_RESP not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else
		  {
		     std::ostringstream oss;
		     oss << "TlmMesh2D::thread : invalid PHASE";
		     throw std::runtime_error(oss.str().c_str());
		  }	       
	       }
	       else if(retval == tlm::TLM_COMPLETED)
	       {
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
	       }
	       else
	       {
		  std::ostringstream oss;
		  oss << "TlmMesh2D::thread : invalid SYNC_ENUM";
		  throw std::runtime_error(oss.str().c_str());
	       }

	       trans->release();

	    } // end of while(true)
	 }      


	 tlm::tlm_sync_enum ip_side_fw(int id,
				       tlm::tlm_generic_payload &trans,
				       tlm::tlm_phase           &phase,
				       sc_core::sc_time         &delay)
	 {
	    // Only write command is supported
	    if(trans.get_command() != tlm::TLM_WRITE_COMMAND)
	    {
	       std::ostringstream oss;
	       oss << "TlmMesh2D::ip_side_fw " << id 
		   << ": only write command is supported";
	       throw std::runtime_error(oss.str().c_str());
	    }
	 
	    if(phase == tlm::BEGIN_REQ)
	    {
	       trans.acquire();
	       _ip_peq[id]->notify(trans, delay);
	    }
	    else if(phase == tlm::END_RESP)
	    {
	       trans.set_response_status(tlm::TLM_OK_RESPONSE);
	       return tlm::TLM_COMPLETED;
	    }
	    else
	    {
	       std::ostringstream oss;
	       oss << "TlmMesh2D::ip_side_fw " << id 
		   << ": got invalid PHASE";
	       throw std::runtime_error(oss.str().c_str());
	    }
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_ACCEPTED;
	 }


	 tlm::tlm_sync_enum ip_side_bw(int id,
				       tlm::tlm_generic_payload &trans,
				       tlm::tlm_phase           &phase,
				       sc_core::sc_time         &delay)
	 {
	    if(phase == tlm::BEGIN_REQ || phase == tlm::END_RESP)
	    {
	       std::ostringstream oss;
	       oss << "TlmMesh2D::ip_side_bw " << id << " got wrong phase";
	       throw std::runtime_error(oss.str().c_str());
	    }

	    _iptxCompleteEvent[id].notify(delay);
	 
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_COMPLETED;
	 }


	 void noc_side_thread(unsigned int agent)
	 {
	    tlm::tlm_generic_payload* trans = 0;
	    tlm::tlm_phase            phase;
	    sc_core::sc_time          delay;
	    tlm::tlm_sync_enum        retval;
	    unsigned int              source = 0;

	    //std::cout << "TlmMesh2D::noc_thread(" << agent << ")" << std::endl;
	    // std::ostringstream oss;
	    // oss << "at_" << agent << "_to_ip.txt";
	    // std::ofstream ofs(oss.str().c_str());

	    while(true)
	    {
	       // Check for pending transactions
	       if((trans = _noc_peq[agent]->get_next_transaction()) == 0)
	       { 
		  wait(_noc_peq[agent]->get_event());
		  trans = _noc_peq[agent]->get_next_transaction();
	       }

	       // ofs << "Agent " << agent << " getting data  "
	       // 	   << sc_core::sc_time_stamp().value() << std::endl;

	       phase = tlm::END_REQ;
	       delay = _cycleTime * 
		  ((trans->get_data_length() / trans->get_streaming_width() +
		    3));

	       retval = _target[agent]->nb_transport_bw(*trans, 
							phase, delay);
	    
	       if(retval != tlm::TLM_COMPLETED)
	       {
		  std::ostringstream oss;
		  oss << "TlmMesh2Dr::noc_thread : Not supporting responses";
		  throw std::runtime_error(oss.str().c_str());
	       }
	    
	       // Forward transaction

	       phase = tlm::BEGIN_REQ;
	       delay = _cycleTime * 
		  ((trans->get_data_length() / trans->get_streaming_width() +
		    2));
	    
	       retval = initSockets[agent]->nb_transport_fw(*trans, phase, delay);
	    
	       if(retval == tlm::TLM_ACCEPTED || retval == tlm::TLM_UPDATED)
	       {
		  if(phase == tlm::BEGIN_REQ)
		  { 
		     wait(_iptxCompleteEvent[agent]);		
		  }
		  else if(phase == tlm::END_REQ)
		  {
		     std::ostringstream oss;
		     oss << "TlmMesh2D::thread : END_REQ not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else if(phase == tlm::BEGIN_RESP)
		  {
		     std::ostringstream oss;
		     oss << "TlmMesh2D::thread : BEGIN_RESP not supported";
		     throw std::runtime_error(oss.str().c_str());
		  }
		  else
		  {
		     std::ostringstream oss;
		     oss << "TlmMesh2D::thread : invalid PHASE";
		     throw std::runtime_error(oss.str().c_str());
		  }	       
	       }
	       else if(retval == tlm::TLM_COMPLETED)
	       {
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
	       }
	       else
	       {
		  std::ostringstream oss;
		  oss << "TlmMesh2D::thread : invalid SYNC_ENUM";
		  throw std::runtime_error(oss.str().c_str());
	       }

	       trans->release();

	    } // end of while(true)
	 }      


	 tlm::tlm_sync_enum noc_side_fw(int id,
					tlm::tlm_generic_payload &trans,
					tlm::tlm_phase           &phase,
					sc_core::sc_time         &delay)
	 {
	    // Only write command is supported
	    if(trans.get_command() != tlm::TLM_WRITE_COMMAND)
	    {
	       std::ostringstream oss;
	       oss << "TlmMesh2D::noc_side_fw " << id 
		   << ": only write command is supported";
	       throw std::runtime_error(oss.str().c_str());
	    }
	 
	    if(phase == tlm::BEGIN_REQ)
	    {
	       trans.acquire();
	       _noc_peq[id]->notify(trans, delay);
	    }
	    else if(phase == tlm::END_RESP)
	    {
	       trans.set_response_status(tlm::TLM_OK_RESPONSE);
	       return tlm::TLM_COMPLETED;
	    }
	    else
	    {
	       std::ostringstream oss;
	       oss << "TlmMesh2D::noc_side_fw " << id 
		   << ": got invalid PHASE";
	       throw std::runtime_error(oss.str().c_str());
	    }
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_ACCEPTED;
	 }


	 tlm::tlm_sync_enum noc_side_bw(int id,
					tlm::tlm_generic_payload &trans,
					tlm::tlm_phase           &phase,
					sc_core::sc_time         &delay)
	 {
	    if(phase == tlm::BEGIN_REQ || phase == tlm::END_RESP)
	    {
	       std::ostringstream oss;
	       oss << "TlmMesh2D::noc_side_bw " << id << " got wrong phase";
	       throw std::runtime_error(oss.str().c_str());
	    }

	    _noctxCompleteEvent[id].notify(delay);
	 
	    trans.set_response_status( tlm::TLM_OK_RESPONSE );
	    return tlm::TLM_COMPLETED;
	 }



	 tlm_utils::simple_initiator_socket_tagged
	 <TlmMesh2D, data_width_g> _init[rows_g * cols_g];
	 tlm_utils::simple_target_socket_tagged
	 <TlmMesh2D, data_width_g> _target[rows_g * cols_g];
	 
	 TlmMeshRouter<data_width_g>* _routers[rows_g * cols_g];

	 tlm_utils::peq_with_get<tlm::tlm_generic_payload>* 
	    _ip_peq[rows_g * cols_g];
	 
	 tlm_utils::peq_with_get<tlm::tlm_generic_payload>* 
	    _noc_peq[rows_g * cols_g];

	 sc_core::sc_event _iptxCompleteEvent[rows_g * cols_g];
	 sc_core::sc_event _noctxCompleteEvent[rows_g * cols_g];

	 sc_core::sc_time _cycleTime;

      };

   }
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:

