/**
 * File:    ServicePanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 26.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.service;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.Rectangle2D;
import java.text.DecimalFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;

import fi.cpu.Settings;
import fi.cpu.data.Chart;
import fi.cpu.data.Configuration;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Process;
import fi.cpu.data.ProcessingElement;
import fi.cpu.data.Service;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;
import fi.cpu.table.ReportTableCellRenderer;
import fi.cpu.table.ReportTableHeaderRenderer;
import fi.cpu.ui.MainWindow;
import fi.cpu.ui.graph.GraphPanel;
import fi.cpu.ui.graph.SimpleChartFactory;


/**
 * A panel fro showing report data of a service.
 */
public class ServicePanel extends JPanel implements Comparable<ServicePanel> {
    private static final Color BACKGROUND;
    private static final Color TEXT_BACKGROUND;
    private static final Color TEXT_COLOR;
    
    protected static Vector<String> COLUMN_NAMES = new Vector<String>();
    protected static final String SERVICE_TOTAL_LABEL;
    protected static final String SERVICE_CHART_TITLE;
    protected static final String CHART_ID_PREFIX = "service";
    protected static final int FRACTION_DIGITS;

    protected Service service;
    protected Configuration config;
	protected Vector<Process> processes;
	protected Map<Integer, ProcessingElement> processMapping;
	protected DefaultTableModel tableModel;
	protected DecimalFormat doubleFormatter;
	protected DecimalFormat intFormatter;
	protected JTable table;
    protected JPopupMenu popupMenu;
	protected int popupRow = 0;
	protected int popupColumn = 0;
    protected Font defaultFont;
    protected Font sumRowFont;
    
    
	static {
		ResourceBundle bundle = MainWindow.bundle;
		COLUMN_NAMES.add(" ");
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_PE_MAP"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_ECOUNT"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_AVG_ETIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_TOT_ETIME"));
		COLUMN_NAMES.add(bundle.getString("REPORT_LABEL_SERVICE_REL_ETIME"));
		SERVICE_TOTAL_LABEL = bundle.getString("REPORT_LABEL_SERVICE_TOTAL");
		SERVICE_CHART_TITLE = bundle.getString("CHART_TITLE_SERVICE");
		
		FRACTION_DIGITS = Integer.parseInt(Settings.getAttributeValue("REPORT_FRACTION_DIGITS"));
		
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_BLUE") );
        TEXT_COLOR = new Color(red, green, blue);
 
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_BLUE") );
        TEXT_BACKGROUND = new Color(red, green, blue);
        
        red = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_BLUE") );
        BACKGROUND = new Color(red, green, blue);
	}

	
	/**
	 * Creates a new ServicePanel.
	 */
	public ServicePanel(Service service) {
    	super();
    	this.service = service;
    	
    	config = MainWindow.getInstance().getConfiguration();
    	processes = new Vector<Process>();
    	processMapping = new HashMap<Integer, ProcessingElement>();
    	config.getServiceModel().addModelListener(new MyServiceModelListener());
    	config.getPeModel().addModelListener(new MyPeModelListener());
    	service.addModelNodeListener(new MyServiceListener());
        
        setBackground(TEXT_BACKGROUND);
        setBorder(BorderFactory.createLineBorder(Color.BLACK));
        setLayout(new GridBagLayout());

        // Create name label
        JLabel name = new JLabel(service.getName());
        Font currentFont = name.getFont();        
        int font_size = Integer.parseInt(Settings.getAttributeValue("PROCESSOR_TEXT_FONT"));
           
        Font newFont = new Font(currentFont.getName(), currentFont.getStyle(), font_size);
        name.setBackground(TEXT_BACKGROUND);
        name.setForeground(TEXT_COLOR);
        name.setFont(newFont);

		defaultFont = getFont();
		sumRowFont = defaultFont.deriveFont(Font.BOLD);
		tableModel = new DefaultTableModel(COLUMN_NAMES, 0);
		
		// Create container for the table
		JPanel tableContainer = new JPanel();
		tableContainer.setBackground(BACKGROUND);
		tableContainer.setLayout(new BorderLayout());

		// Create the table view
		table = new JTable(tableModel);
		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		table.getTableHeader().setReorderingAllowed(false);
		table.setDefaultRenderer(Object.class, new ReportTableCellRenderer(defaultFont, sumRowFont));
		table.setDefaultEditor(Object.class, null);
		table.addMouseListener(new MyMouseListener());
		
		// Set column widths to be large enough to show full titles
		TableColumn column = null;
		FontMetrics fm = null;
		Graphics g = getGraphics();
		for (int i=0; i<COLUMN_NAMES.size(); ++i)
		{
			String columnTitle = null;
			if (i == 0)
			{
				columnTitle = SERVICE_TOTAL_LABEL;
				fm = getFontMetrics(sumRowFont);
			}
			else 
			{
				columnTitle = COLUMN_NAMES.get(i);
				fm = getFontMetrics(defaultFont);
			}
			Rectangle2D bounds = fm.getStringBounds(columnTitle, g);
			int width = (int)bounds.getWidth();

			column = table.getColumnModel().getColumn(i);
			column.setPreferredWidth(width+8); // a little bit extra for borders, empty space etc.
		}
		
		// Configure the table header
		DefaultTableCellRenderer headerRenderer = new ReportTableHeaderRenderer();
		table.getTableHeader().setDefaultRenderer(headerRenderer);
		table.getTableHeader().setBackground(BACKGROUND);

		// Add items to panel
		JScrollPane scrollPane = new JScrollPane(table, ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
		ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        
        scrollPane.getViewport().setBackground(BACKGROUND);
        scrollPane.setBorder(null);
        scrollPane.getViewport().setBorder(null);
        tableContainer.add(scrollPane, BorderLayout.CENTER);
        
		popupMenu = new JPopupMenu();
		JMenuItem showChartItem = new JMenuItem(MainWindow.bundle.getString("SHOW_GRAPH"));
		showChartItem.addActionListener(new MyActionListener());
		popupMenu.add(showChartItem);        
        
        add(name, new GridBagConstraints(
        		0, 0, GridBagConstraints.REMAINDER, 1,
        		0.0, 0.0, GridBagConstraints.NORTHWEST, GridBagConstraints.NONE,
        		new Insets(2, 2, 2, 2), 0, 0));

        add(tableContainer, new GridBagConstraints(
        		0, 1, GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER,
        		1.0, 1.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));
        
        initPanel();
	}
	
	
	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		processes.clear();
		
        Iterator<ModelNode> sIter = config.getServiceModel().getChildren(service).iterator();

        while (sIter.hasNext()) {
        	ModelNode node = sIter.next();
        	if (!(node instanceof Process)) {
        		continue;
        	}
        	Process p = (Process) node;
        	processes.add(p);
        	processMapping.put(p.getId(), null);
        }
        
        // Sort processes
		Collections.sort(processes);

		// clear the table
		tableModel.setRowCount(0);
		
		// add process rows
		for (int i=0; i<processes.size(); ++i) {			
			Process p = processes.get(i);

			ProcessingElement pe = processMapping.get(p.getId());
			String peName = "";
			if (pe != null) {
				peName = pe.getName();
			}
			
			Object[] rowData = {p.getName(), peName, new Integer(p.getExecCount()),
					new Double(p.getAvgExecTime()), new Double(p.getTotExecTime()), new Double(0)};
			tableModel.addRow(rowData);
		}
		
		// add total row
		Object[] sumRow = {SERVICE_TOTAL_LABEL, "",
				new Integer(service.getExecCount()),
				new Double(service.getAvgExecTime()),
				new Double(service.getTotExecTime()),
				new Double(0)};
		tableModel.addRow(sumRow);
		
		updateDataTable();
		updateMappingValues();
	}

	
	/**
	 * Updates the data in the table.
	 */
	protected void updateDataTable() {		
		// update process rows
		for (int i=0; i<processes.size(); ++i) {			
			Process p = processes.get(i);

			ProcessingElement pe = processMapping.get(p.getId());
			String peName = "";
			if (pe != null) {
				peName = pe.getName();
			}
			
			Object[] rowData = {p.getName(), peName, new Integer(p.getExecCount()),
					new Double(p.getAvgExecTime()), new Double(p.getTotExecTime()), new Double(0)};
			
			for (int j=0; j<rowData.length; ++j) {
				tableModel.setValueAt(rowData[j], i, j);
			}
		}
		
		// update total row
		Object[] sumRow = {SERVICE_TOTAL_LABEL, "",
				new Integer(service.getExecCount()),
				new Double(service.getAvgExecTime()),
				new Double(service.getTotExecTime()),
				new Double(0)};

		for (int j=0; j<sumRow.length; ++j) {
			tableModel.setValueAt(sumRow[j], tableModel.getRowCount()-1, j);
		}
		
		updateRelativeValues();
				
		// update service charts
		int threadRowIndex = table.getRowCount()-1;

		for (int i=2; i<table.getColumnCount(); ++i) {
			StringBuilder chartID = new StringBuilder();
			chartID.append(CHART_ID_PREFIX);
			chartID.append(service.getId());
			chartID.append(":");
			chartID.append(table.getColumnName(i));
			
			Number value = (Number)table.getValueAt(threadRowIndex, i);
			MainWindow.getInstance().addValueToChart(
					null, chartID.toString(), value.doubleValue());
		}

		// refresh the view
		revalidate();
		repaint();
	}
	
	
	/**
	 * Updates the column with thread relative values.
	 */
	private void updateRelativeValues() {
		Double threadTotETime = (Double)tableModel.getValueAt(tableModel.getRowCount()-1, 4);
		Double sum = 0.0;
		
		for (int i=0; i<processes.size(); ++i) {
			Process p = processes.get(i);
			Double tRel = new Double(0);
			
			// do not divide by 0
			if (threadTotETime.doubleValue() != 0) {
				tRel = new Double((p.getTotExecTime() / threadTotETime.doubleValue()) * 100);
			}
			tableModel.setValueAt(new Double(tRel), i, 5);
			sum += tRel;
		}
		tableModel.setValueAt(new Double(sum), tableModel.getRowCount()-1, 5);
	}
	
	
	/**
	 * Updates the column with mapping info
	 */
	protected void updateMappingValues() {
        // Update mapping info
        List<ModelNode> pList = config.getPeModel().getNodesOfType(Process.class);
        for (int i=0; i<pList.size(); ++i) {
        	Process p = (Process) pList.get(i);
        	if (processMapping.containsKey(p.getId())) {
        		List<ModelNode> path = config.getPeModel().getPathFromRoot(p);
        		ModelNode peNode = path.get(1);
        		if (peNode instanceof ProcessingElement) {
        			ProcessingElement pe = (ProcessingElement) peNode;
        			processMapping.put(p.getId(), pe);
        		}
        	}
        }
        
		for (int i=0; i<processes.size(); ++i) {			
			Process p = processes.get(i);

			ProcessingElement pe = processMapping.get(p.getId());
			String peName = "";
			if (pe != null) {
				peName = pe.getName();
			}
			
			tableModel.setValueAt(peName, i, 1);
		}
	}
	
	
	/**
	 * Implements the Compararable-interface
	 */
    public int compareTo(ServicePanel o) {
    	return service.compareTo(o.service);
    }
	
	
	/**
	 * Listener for MouseEvents that may trigger the popup menu.
	 */
	private class MyMouseListener extends MouseAdapter {
		public void mousePressed(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		public void mouseReleased(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		private void showPopupIfTriggered(MouseEvent e) {
			if (e.isPopupTrigger()) {
				Point p = e.getPoint();
				popupRow = table.rowAtPoint(p);
				popupColumn = table.columnAtPoint(p);
				if (popupColumn > 1 && popupRow >= 0) {
					popupMenu.show(e.getComponent(), e.getX(), e.getY());
				}
			}			
		}
	}
	
	
	/**
	 * Listener for actions from the popup menu.
	 */
	private class MyActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			StringBuilder title = new StringBuilder();
			StringBuilder chartID = new StringBuilder();
			
			if (popupRow == table.getRowCount()-1) {
				title.append(SERVICE_CHART_TITLE);
				title.append(" ");
				title.append(service.getName());
				
				chartID.append(CHART_ID_PREFIX);
				chartID.append(service.getId());
				
			} else {
				String pName = (String)table.getValueAt(popupRow, 0);
				title.append(pName);
				chartID.append(pName);
			}
			
			title.append(": ");
			title.append(table.getColumnName(popupColumn));
			int historySize = 0;
			chartID.append(":");
			chartID.append(table.getColumnName(popupColumn));
			
			try {
				String history = System.getProperty("history");
				if (history != null) {
					historySize = Integer.parseInt(history);
				}
			} catch (NumberFormatException nfe) {
				// do nothing
			}
			
			// Create the chart and add it to main window chart panel
			GraphPanel panel = new GraphPanel();
			Chart chart = new Chart(chartID.toString(), SimpleChartFactory.LINECHART_TYPE,
					title.toString(), MainWindow.bundle.getString("XAXIS"), "", 0.0, 0.0,
					historySize, true, false);
			panel.setChart(chart);
			MainWindow.getInstance().addChart(panel, 10, 10, 200, 300);
		}
	}

	
	/**
	 * Listener for events from the represented service.
	 */
	private class MyServiceListener implements ModelNodeListener {
		public void modelNodeChanged(ModelNodeEvent e) {
			if (e.getEventId() == Service.DATA_CHANGED_EVENT) {
				updateDataTable();
			}
		}
	}
	
	
    /**
     * Listener for service model events.
     */
    private class MyServiceModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		if (parent != null && parent.equals(service)) {
    			// Service's children changed.
    			initPanel();
    		}
    	}
    	public void nodeRemoved(ModelEvent e) {
    		ModelNode node = e.getNode();
    		ModelNode oldParent = e.getOldParent();
    		
    		if (node != null && node.equals(service)) {
    			// Service was removed.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getServiceModel().removeModelListener(this);
    		} else if (oldParent != null && oldParent.equals(service)) {
    			// Service's children changed.
    			initPanel();
    		}
    	}
    	public void nodeMoved(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		ModelNode oldParent = e.getOldParent();
    		
    		if ((parent != null && parent.equals(service))
    				|| (oldParent != null && oldParent.equals(service))) {
    			// Service's children changed.
    			initPanel();    			
    		}
    	}
    	public void structureChanged(ModelEvent e) {
    		ModelNode node = e.getNode();
    		
    		if (node != null && node.equals(service)) {
    			// Structure changed starting from this node.
    			initPanel();
    		} else if (config.getPeModel().isChildOf(service, node)) {
    			// Service is part of the changed structure.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		}
    	}
    }
    
    
    /**
     * Listener for pe model events.
     */
    private class MyPeModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
			updateMappingValues();
    	}
    	public void nodeRemoved(ModelEvent e) {
			updateMappingValues();
    	}
    	public void nodeMoved(ModelEvent e) {
			updateMappingValues();
    	}
    	public void structureChanged(ModelEvent e) {
			updateMappingValues();
    	}
    }
}
