/**
 * CVS:
 * $Id: ProcessorPanel.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    ProcessorPanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 1.12.2006
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.report;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;

import fi.cpu.Settings;
import fi.cpu.data.Configuration;
import fi.cpu.data.ModelNode;
import fi.cpu.data.ProcessingElement;
import fi.cpu.data.Thread;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;
import fi.cpu.ui.MainWindow;


/**
 * A panel which contains the report of a PE.
 */
public class ProcessorPanel extends JPanel implements Comparable<ProcessorPanel> {
    public static final Color TEXT_BACKGROUND;
    public static final Color TEXT_COLOR;

    protected ProcessingElement pe;
    protected Configuration config;
    private JPanel threadPanelContainer;
    protected MyThreadListener threadListener;
    protected Map<String, ThreadPanel> threadPanels;
    
    static {
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_COLOR_BLUE") );
        TEXT_COLOR = new Color(red, green, blue);
 
        red = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("PROCESSOR_TEXT_BACKGROUD_COLOR_BLUE") );
        TEXT_BACKGROUND = new Color(red, green, blue);    	
    }
    
    
    /**
     * Creates a new ProcessorPanel.
     */
    public ProcessorPanel(ProcessingElement pe) {
    	super();
    	this.pe = pe;
    	
    	config = MainWindow.getInstance().getConfiguration();
    	
    	threadListener = new MyThreadListener();
        threadPanels = new HashMap<String, ThreadPanel>();
        config.getPeModel().addModelListener(new MyModelListener());
        
        setBackground(TEXT_BACKGROUND);
        setBorder(BorderFactory.createLineBorder(Color.BLACK));
        setLayout(new GridBagLayout());

        threadPanelContainer = new JPanel();
        threadPanelContainer.setLayout( new BoxLayout(threadPanelContainer, BoxLayout.Y_AXIS) );
        threadPanelContainer.setPreferredSize(new Dimension(250, 140));

        JLabel name = new JLabel(pe.getName());
        Font currentFont = name.getFont();        
        int font_size = Integer.parseInt(Settings.getAttributeValue("PROCESSOR_TEXT_FONT"));
           
        Font newFont = new Font(currentFont.getName(), currentFont.getStyle(), font_size);
        name.setBackground(TEXT_BACKGROUND);
        name.setForeground(TEXT_COLOR);
        name.setFont(newFont);
        
        add(name, new GridBagConstraints(
        		0, 0, GridBagConstraints.REMAINDER, 1,
        		0.0, 0.0, GridBagConstraints.NORTHWEST, GridBagConstraints.NONE,
        		new Insets(2, 2, 2, 2), 0, 0));

        add(threadPanelContainer, new GridBagConstraints(
        		0, 1, GridBagConstraints.REMAINDER, 1,
        		1.0, 0.5, GridBagConstraints.NORTHWEST, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));

        initPanel();
    }


	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		threadPanels.clear();
		
        Iterator<ModelNode> tIter = config.getPeModel().getChildren(pe).iterator();

        while (tIter.hasNext()) {
        	ModelNode node = tIter.next();
        	if (!(node instanceof Thread)) {
        		continue;
        	}
        	Thread t = (Thread) node;
        	t.addModelNodeListener(threadListener);
        	threadPanels.put(t.getId(), new ThreadPanel(t));
        }
        drawThreadPanels();
	}

	
    /**
     * Draws thread panels.
     */
    protected void drawThreadPanels() {
    	// Remove old panels
        threadPanelContainer.removeAll();
        
		// Sort panels
		ArrayList<ThreadPanel> pList = new ArrayList<ThreadPanel>(threadPanels.values());
		Collections.sort(pList);
		
		// Add new panels
        Iterator<ThreadPanel> iter = pList.iterator();
        while(iter.hasNext()) {
        	ThreadPanel panel = iter.next();
        	threadPanelContainer.add(panel);
        }
        
        revalidate();
        repaint();
    }

    
	/**
	 * Implements the Compararable-interface
	 */
    public int compareTo(ProcessorPanel o) {
    	return pe.compareTo(o.pe);
    }

    
    /**
     * Listener for ThreadEvents from contained threads.
     */
    private class MyThreadListener implements ModelNodeListener {
    	public void modelNodeChanged(ModelNodeEvent e) {
    		if (e.getEventId() == Thread.PRIORITY_CHANGED_EVENT) {
        		drawThreadPanels();    			
    		}
    	}
    }
    
    
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		if (parent != null && parent.equals(pe)) {
    			// PE's children changed.
    			initPanel();
    		}
    	}
    	public void nodeRemoved(ModelEvent e) {
    		ModelNode node = e.getNode();
    		ModelNode oldParent = e.getOldParent();
    		
    		if (node != null && node.equals(pe)) {
    			// PE was removed.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		} else if (oldParent != null && oldParent.equals(pe)) {
    			// PE's children changed.
    			initPanel();
    		}
    	}
    	public void nodeMoved(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		ModelNode oldParent = e.getOldParent();
    		
    		if ((parent != null && parent.equals(pe))
    				|| (oldParent != null && oldParent.equals(pe))) {
    			// PE's children changed.
    			initPanel();    			
    		}
    	}
    	public void structureChanged(ModelEvent e) {
    		ModelNode node = e.getNode();
    		
    		if (node != null && node.equals(pe)) {
    			// Structure changed starting from this node.
    			initPanel();
    		} else if (config.getPeModel().isChildOf(pe, node)) {
    			// PE is part of the changed structure.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		}
    	}
    }
}
