/**
 * File:    ThreadPanel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 8.3.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.ui.ctrl;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.border.TitledBorder;

import fi.cpu.Settings;
import fi.cpu.data.Configuration;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Process;
import fi.cpu.data.Thread;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.event.ModelNodeEvent;
import fi.cpu.event.ModelNodeListener;
import fi.cpu.ui.MainWindow;
import fi.cpu.ui.ScrollablePanel;
import fi.cpu.ui.ctrl.ProcessIcon.State;


/**
 * The ThreadPanel represents a thread in the control view.
 */
public class ThreadPanel extends JPanel implements Comparable<ThreadPanel> {
    public static final Color BACKGROUND;
    public static final Color BORDER_COLOR;

    protected ThreadPanel self;
    protected Configuration config;
    protected Thread thread;
    protected TitledBorder border;
    protected JPopupMenu popupMenu;
    protected Map<Integer, ProcessIcon> processIcons;
    protected ScrollablePanel processIconContainer;
    protected JScrollPane processScrollPane;

    static {
        int red = 0;
        int green = 0;
        int blue = 0;
        
        red = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("THREAD_BACKGROUD_COLOR_BLUE") );

        BACKGROUND = new Color(red, green, blue);

        red = Integer.parseInt( Settings.getAttributeValue("THREAD_TEXT_COLOR_RED") );
        green = Integer.parseInt( Settings.getAttributeValue("THREAD_TEXT_COLOR_GREEN") );
        blue = Integer.parseInt( Settings.getAttributeValue("THREAD_TEXT_COLOR_BLUE") );
        
        BORDER_COLOR = new Color(red, green, blue);
    }

    
	/**
	 * Creates a new ThreadPanel.
	 * @param thread The thread model object.
	 */
	public ThreadPanel(Thread thread) {
		super();
		this.self = this;
		this.thread = thread;
        processIcons = new HashMap<Integer, ProcessIcon>();
        config = MainWindow.getInstance().getConfiguration();

        border = BorderFactory.createTitledBorder(BorderFactory.createLineBorder( BORDER_COLOR ),
        		createBorderTitle());
        
        setBorder(border);
        setBackground(BACKGROUND);
        setLayout(new GridBagLayout());

        processIconContainer = new ScrollablePanel();
        processIconContainer.setLayout(null);
        processIconContainer.setBackground(BACKGROUND);
        processIconContainer.setBorder(null);
        
        processScrollPane = new JScrollPane(processIconContainer,
        		ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
        		ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);

        processScrollPane.getViewport().setBackground(BACKGROUND);
        processScrollPane.setBorder(null);
        
        this.add(processScrollPane, new GridBagConstraints(
        		0, 0, GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER,
        		1.0, 1.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH,
        		new Insets(0, 0, 0, 0), 0, 0));
        
        // Create the popup menu
		popupMenu = new JPopupMenu();
		JMenuItem changePriorityItem = new JMenuItem(MainWindow.bundle.getString("MENU_CHANGE_PRIORITY"));
		changePriorityItem.addActionListener(new MyActionListener());
		popupMenu.add(changePriorityItem);
		
        // Add listeners
        MyMouseListener mouseListener = new MyMouseListener();                
        addMouseListener(mouseListener);
        processIconContainer.addMouseListener(mouseListener);
        processScrollPane.addMouseListener(mouseListener);
        config.getPeModel().addModelListener(new MyModelListener());
        
        thread.addModelNodeListener(new MyThreadListener());
        addComponentListener(new MyComponentListener());

        initPanel();
	}

	
	/**
	 * Initializes the panel.
	 */
	public void initPanel() {
		processIcons.clear();
		
        Iterator<ModelNode> pIter = config.getPeModel().getChildren(thread).iterator();

        while (pIter.hasNext()) {
        	ModelNode node = pIter.next();
        	if (!(node instanceof Process)) {
        		continue;
        	}
        	Process p = (Process) node;
        	processIcons.put(p.getId(), new ProcessIcon(p));
        }
        
        drawProcesses();		
	}
		
	
	/**
	 * @return The thread model object.
	 */
	public Thread getThread() {
		return thread;
	}

	
    /**
     * @return The title for the panel border.
     */
    protected String createBorderTitle() {
        StringBuilder title = new StringBuilder(" ");
        title.append(MainWindow.bundle.getString("THREAD_TITLE_THREAD"));
        title.append(" ");
        title.append(thread.getName());
        title.append(" - ");
        title.append(MainWindow.bundle.getString("THREAD_TITLE_PRIORITY"));
        title.append(" ");
        title.append(thread.getPriority());
        return title.toString();
    }
    
    
    /**
     * Adds a ProcessIcon to this panel.
     * @param icon
     */
    public void addProcessIcon(ProcessIcon icon) {
		processIcons.put(icon.getProcess().getId(), icon);
		drawProcesses();
    }

    
    /**
     * Removes a ProcessIcon from this panel.
     * @param icon
     */
    public void removeProcessIcon(ProcessIcon icon) {
    	processIcons.remove(icon.getProcess().getId());
    	drawProcesses();
    }
    
    
    /**
     * @return Returns a ProcessIcon corresponding to given process id.
     */
    public ProcessIcon getProcessIcon(Integer id) {
    	return processIcons.get(id);
    }
    
    
    /**
     * Commits changes in process locations.
     */
    public void commitProcessMoves() {
    	List<ProcessIcon> pList = new ArrayList<ProcessIcon>(processIcons.values());
    	Iterator<ProcessIcon> iter = pList.iterator();
    	while (iter.hasNext()) {
    		ProcessIcon icon = iter.next();
    		State state = icon.getState();
    		if (state == State.REMOVED) {
    			removeProcessIcon(icon);
    			
    		} else if (state == State.MOVED) {
    			icon.setState(State.NORMAL);
    			config.getPeModel().moveNode(icon.getProcess(), thread);
    		}
    	}
    }
    
	
    /**
     * Draws processes.
     */
    protected void drawProcesses() {
        processIconContainer.removeAll();
        
        Point place = null;
        
        if(ProcessIcon.SQUARE_SHAPE) {
            place = new Point( 5, 20 );
        } else {
            place = new Point( 10, 20 );
        }
        
        double vpHeight = processScrollPane.getViewport().getSize().getHeight();
        boolean first = true;
        
		// Sort icons
		ArrayList<ProcessIcon> pList = new ArrayList<ProcessIcon>(processIcons.values());
		Collections.sort(pList);
        
        Iterator iter = pList.iterator();        
        while(iter.hasNext()) {
        	
            ProcessIcon process = (ProcessIcon)iter.next();
            int pHeight = process.getHeight();
            
            if( (  place.y + pHeight ) > vpHeight && !first) {
            	place.y = 20;
            	place.x = place.x +process.getWidth() + 5;
            }
            
            process.setLocation( place.x, place.y );
            place.y = place.y + 5 + pHeight;
            first = false;
            
            processIconContainer.add( process );
        }
        
        revalidate();
        repaint();
    }
    
    
	/**
	 * Implements the Compararable-interface
	 */
    public int compareTo(ThreadPanel o) {
    	return thread.compareTo(o.thread);
    }

    
    /**
     * Listener for ComponentEvents.
     */
    private class MyComponentListener extends ComponentAdapter {
    	public void componentResized(ComponentEvent e) {
    		drawProcesses();
    	}
    }
    
	
	/**
	 * Listener for MouseEvents.
	 */
	private class MyMouseListener extends MouseAdapter {	
		public void mousePressed(MouseEvent e) {
			showPopupIfTriggered(e);
		}
		
		public void mouseReleased(MouseEvent e) {
			showPopupIfTriggered(e);
		}

		private void showPopupIfTriggered(MouseEvent e) {
			if (e.isPopupTrigger()) {
				popupMenu.show(e.getComponent(), e.getX(), e.getY());
			}			
		}
	}

	
	/**
	 * Listener for actions from popup menu.
	 */
	private class MyActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			boolean legalValueGiven = false;
			String message = null;
			String oldValue = ""+thread.getPriority();
			int priority = -1;
						
			while (!legalValueGiven) {
				Object value = JOptionPane.showInputDialog(self, message,
						MainWindow.bundle.getString("PRIORITY_DIALOG_TITLE"),
						JOptionPane.PLAIN_MESSAGE, null, null, oldValue);
				if (value == null) {
					return; // cancel pressed
				}
				try {
					priority = Integer.parseInt((String)value);
				} catch (NumberFormatException nfe) {
					// wasn't integer.
					oldValue = (String)value;
					if (message == null) {
						message = MainWindow.bundle.getString("PRIORITY_DIALOG_NO_INT");
					} else {
						message = message+"!";
					}
					continue; // ask again
				}
				legalValueGiven = true;
			}

			if (legalValueGiven) {
				thread.setPriority(priority);
				MainWindow.getInstance().sendMappingString();
			}
		}
	}
	
	
	/**
	 * Listener for ThreadEvents.
	 */
	private class MyThreadListener implements ModelNodeListener {
		public void modelNodeChanged(ModelNodeEvent e) {
			if (e.getEventId() == Thread.PRIORITY_CHANGED_EVENT) {
				border.setTitle(createBorderTitle());
				repaint();	
			}			
		}
	}
	
	
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		if (parent != null && parent.equals(thread)) {
    			// Thread's children changed.
    			initPanel();
    		}
    	}
    	public void nodeRemoved(ModelEvent e) {
    		ModelNode node = e.getNode();
    		ModelNode oldParent = e.getOldParent();
    		
    		if (node != null && node.equals(thread)) {
    			// Thread was removed.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		} else if (oldParent != null && oldParent.equals(thread)) {
    			// Thread's children changed.
    			initPanel();
    		}
    	}
    	public void nodeMoved(ModelEvent e) {
    		ModelNode parent = e.getCurrentParent();
    		ModelNode oldParent = e.getOldParent();
    		
    		if ((parent != null && parent.equals(thread))
    				|| (oldParent != null && oldParent.equals(thread))) {
    			// Thread's children changed.
    			initPanel();    			
    		}
    	}
    	public void structureChanged(ModelEvent e) {
    		ModelNode node = e.getNode();
    		
    		if (node != null && node.equals(thread)) {
    			// Structure changed starting from this node.
    			initPanel();
    		} else if (config.getPeModel().isChildOf(thread, node)) {
    			// Thread is part of the changed structure.
    			// Stop listening events, because this panel
    			// will be removed.
    			config.getPeModel().removeModelListener(this);
    		}
    	}
    }
}
