/**
 * CVS:
 * $Id: SignalsTableModel.java 1399 2010-08-26 13:56:45Z lehton87 $
 * 
 * File:    SignalsTableModel.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 22.11.2006
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.table;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import fi.cpu.data.Configuration;
import fi.cpu.data.Model;
import fi.cpu.data.ModelNode;
import fi.cpu.data.Process;
import fi.cpu.data.ProcessingElement;
import fi.cpu.data.Thread;
import fi.cpu.event.ModelEvent;
import fi.cpu.event.ModelListener;
import fi.cpu.ui.MainWindow;


/**
 * SignalsTableModel provides model information for
 * the signals table.
 */
public class SignalsTableModel extends AbstractTableModel {
	private static final String TABLE_TITLE = MainWindow.bundle.getString("SIGNAL_LABEL_TITLE");

	private Configuration config;
	/** Maps indices in the table to process ids */
	private HashMap<Integer, Integer> ind2pidMap;
	/** Maps process ids to indices in the table */
	private HashMap<Integer, Integer> pid2indMap;
	/** Maps process ids to process names */
	private HashMap<Integer, String> pid2nameMap;
	/** Maps source pids to map containing destination pids and signal values. */
	private HashMap< Integer, HashMap<Integer, Integer> > signalValues;
	
	
	/**
	 * Creates new SignalsTableModel.
	 */
	public SignalsTableModel(Configuration config) {
		super();
		this.config = config;
		config.getPeModel().addModelListener(new MyModelListener());

		ind2pidMap = new HashMap<Integer, Integer>();
		pid2indMap = new HashMap<Integer, Integer>();
		pid2nameMap = new HashMap<Integer, String>();
		signalValues = new HashMap< Integer, HashMap<Integer, Integer> >();
		
		initProcesses();
	}

	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	public int getColumnCount() {
		return pid2nameMap.size()+1; // also header column
	}
	
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	public int getRowCount() {
		return ind2pidMap.size();
	}

	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getColumnName(int)
	 */
	public String getColumnName(int column) {
		if (column == 0) {
			// get the left most column name
			return TABLE_TITLE;
		} else {
			// get one of the other column names
			Integer pid = ind2pidMap.get(column-1);
			return pid2nameMap.get(pid);
		}
	}
	
	
	/* (non-Javadoc)
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	public Object getValueAt(int rowIndex, int columnIndex) {
		Integer srcPid = ind2pidMap.get(rowIndex);
		
		if (columnIndex == 0) {
			// get the value in the left most column,
			// i.e. name of the process
			return pid2nameMap.get(srcPid);
		} else {
			// get one of the other values
			Integer destPid = ind2pidMap.get(columnIndex-1);
			
			HashMap<Integer, Integer> map = signalValues.get(srcPid);
			if (map == null) {
				return new Integer(0);
			}
			
			Object val = map.get(destPid); 
			if (val == null) {
				val = new Integer(0);
			}
			return val;
		}
	}
	
	
	/**
	 * Sets the signal value.
	 * @param sourcePid Id of the source process.
	 * @param destPid   Id of the destination process.
	 * @param newValue  The signal value.
	 */
	public void setValue(Integer sourcePid, Integer destPid, int newValue) {
		String sourceName = pid2nameMap.get(sourcePid);
		String destName = pid2nameMap.get(destPid);

		if (sourceName == null || destName == null) {
			// unknown id
			return; 
		}
		
		// Get the inner map. If doesn't exist yet, create.
		HashMap<Integer, Integer> destMap = signalValues.get(sourcePid);
		if (destMap == null) {
			destMap = new HashMap<Integer, Integer>();
			signalValues.put(sourcePid, destMap);
		}
		
		// Set the value
		destMap.put(destPid, newValue);
		fireTableCellUpdated(pid2indMap.get(sourcePid), pid2indMap.get(destPid)+1);
	}

	
	/**
	 * Initializes the table
	 */
	protected void initProcesses() {
		HashMap<Integer, Integer> newInd2pidMap = new HashMap<Integer, Integer>();
		HashMap<Integer, Integer> newPid2indMap = new HashMap<Integer, Integer>();
		HashMap<Integer, String> newPid2nameMap = new HashMap<Integer, String>();
		
		Model peModel = config.getPeModel();
		Iterator<ModelNode> peIter = peModel.getChildren(peModel.getModelRoot()).iterator();
		int processCount = 0;
		
		// for all processors
		while (peIter.hasNext()) {
			ModelNode peNode = peIter.next();
			if (!(peNode instanceof ProcessingElement)) {
				continue;
			}
			ProcessingElement pe = (ProcessingElement) peNode;
			Iterator<ModelNode> tIter = peModel.getChildren(pe).iterator();
			
			// for all threads
			while (tIter.hasNext()) {
				ModelNode tNode = tIter.next();
				if (!(tNode instanceof Thread)) {
					continue;
				}
				Thread thread = (Thread) tNode;
				Iterator<ModelNode> pIter = peModel.getChildren(thread).iterator();
				List<Process> pList = new ArrayList<Process>();
				
				// for all processes
				while(pIter.hasNext()) {
					ModelNode pNode = pIter.next();
					if (!(pNode instanceof Process)) {
						continue;
					}
					Process process = (Process) pNode;
					pList.add(process);
				}
				
				Collections.sort(pList);
				
				// Add sorted processes
				for (int i=0; i<pList.size(); ++i) {
					Process process = pList.get(i);
					Integer pid = process.getId();
					
					// Add process info to new maps
					newInd2pidMap.put(new Integer(processCount), pid);
					newPid2indMap.put(pid, new Integer(processCount));
					newPid2nameMap.put(pid, process.getName());
					
					// Remove possible old info
					pid2nameMap.remove(pid);
					++processCount;
				}
			}
		}
		
		// If there are still processes left in the old map,
		// those don't exist anymore. Clear related values.
		Iterator<Integer> pidIter = pid2nameMap.keySet().iterator();
		while (pidIter.hasNext()) {
			Integer pid = pidIter.next();
			
			// Clear values when source
			signalValues.remove(pid);
			
			// Clear values when destination
			Iterator< HashMap<Integer, Integer> > mapIter = signalValues.values().iterator();
			while (mapIter.hasNext()) {
				HashMap<Integer, Integer> map = mapIter.next();
				if (map != null) {
					map.remove(pid);
				}
			}
		}
		
		ind2pidMap = newInd2pidMap;
		pid2indMap = newPid2indMap;
		pid2nameMap = newPid2nameMap;
		fireTableStructureChanged();
	}
	
	
    /**
     * Listener for ModelEvents.
     */
    private class MyModelListener implements ModelListener {
    	public void nodeInserted(ModelEvent e) {
			initProcesses();
    	}
    	public void nodeRemoved(ModelEvent e) {
			initProcesses();
    	}
    	public void nodeMoved(ModelEvent e) {
			initProcesses();
    	}
    	public void structureChanged(ModelEvent e) {
			initProcesses();
    	}
    }
}
