/**
 *
 * @file buffer.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: buffer.cc 1942 2011-08-17 07:34:03Z lehton87 $
 *
 *
 */


#include "buffer.hh"

#include <stdexcept>
#include <iostream>
#include <sstream>
#include <climits>

namespace sctg
{
  
   Buffer::Buffer(unsigned long int rxBufferSize,
		  unsigned long int txBufferSize,
		  sctg::Configuration& config)
      :
      _rxBufferSize(rxBufferSize),
      _txBufferSize(txBufferSize),
      _rxBufferUsed(0),
      _txBufferUsed(0),
      _config(config),
      _usePackets(false)
   {      
   }
  
   Buffer::~Buffer()
   {
//       for(std::map<unsigned long int, tgToken>::iterator iter = 
// 	     _expectedTokens.begin();
// 	  iter != _expectedTokens.end(); ++iter)
//       {
// 	 std::cout << "Unreceived token " << (*iter).second.id << std::endl;
//       }

      while(!_txQueue.empty())
      {
	 if(_txQueue.front()->data) { delete _txQueue.front()->data; }
	 delete _txQueue.front();
	 _txQueue.pop();
      }
   }

   void Buffer::usePackets(bool use)
   {
      _usePackets = use;
   }
   
   void Buffer::rxPutPacket(tgPacket* packet)
   {
//       std::cout << "CH Got packet to receive side "
// 		<< sc_core::sc_time_stamp().value()
// 		<< std::endl;

      

      // minimun packet size is four bytes
      if(packet->size < 4)
      {
	 std::cout << "rxPutPacket: got under four byte packet" << std::endl;
	 throw std::runtime_error("rxPutPacket: got under four byte packet");
      }
      
      // Confirm that packet fits to buffer
      if(packet->size > rxSpaceLeft())
      {
	 std::cout << "rxPutPacket: Packet doesn't fit to receive buffer"
		   << std::endl;
	 throw std::runtime_error
	    ("rxPutPacket: Packet doesn't fit to receive buffer");
      }                  
     
      // Get id from packet, used to identify packet's target
      unsigned int id = *(reinterpret_cast<unsigned int*>(packet->data));      
 
      // Check that we're expecting it
      if(_expectedTokens.find(id) == _expectedTokens.end())
      {
	 std::ostringstream oss;
	 oss << "rxPutPacket: received unexpected packet with id "
	     << id << " and size " << packet->size;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      
      // Add its size to incoming buffer
      if(_incomingData.find(id) == _incomingData.end())
      {
	 _incomingData[id] = packet->size;
      }
      else
      {
	 _incomingData[id] += packet->size;
      }
      // Update free space counter
      _rxBufferUsed += packet->size;    
      _measurements.rxBytes += packet->size;      


      packet->source = _expectedTokens[id].source;
      packet->return_port = _expectedTokens[id].returnPort;
      packet->dstPort = _expectedTokens[id].destination;
      packet->type = _expectedTokens[id].type;
      packet->packets = _expectedTokens[id].packets;
      packet->respSize = _expectedTokens[id].respSize;
      packet->burstSize = _expectedTokens[id].burstSize;

      if(packet->type == CACHE_MISS && !_usePackets)
      {
// 	 std::cout << "got CACHE packet size " << packet->size 
// 		   << " respSize " << packet->respSize << std::endl;
	 _receivedCaches.push(packet);
	 _rxCacheEvent.notify(sc_core::SC_ZERO_TIME);
      }      
      else if(_usePackets)
      {	 
	 _receivedPackets.push(packet);
	 _rxPacketEvent.notify(sc_core::SC_ZERO_TIME);
      }

      if(_config.getPacketStream() && packet->type != CACHE_MISS)
      {
	 **_config.getPacketStream() 
	    << sc_core::sc_time_stamp().value()
	    << ";" << _expectedTokens[id].id 
	    << ";" << _expectedTokens[id].source
	    << ";" << _expectedTokens[id].destination
	    << ";" << packet->size 
	    << std::endl;
      }

      // Delete packet
      if(!_usePackets && packet->type != CACHE_MISS)
      {
	 delete packet->data;
	 delete packet;
      }


      // If there is enough data received for token remove temps and
      // inform event

      if(_incomingData[id] >= _expectedTokens[id].size)
      {
	 // Update free space counter if received more than expected
	 //  Some networks migh give us more bytes than needed if
	 //  using for example fixed size packets and network
	 //  interface doesn't remove extra bytes... just to be sure.
	 _rxBufferUsed -= (_incomingData[id] - _expectedTokens[id].size);
	 _expectedTokens[id].timeReceived = sc_core::sc_time_stamp();
	 
	 if(_config.getTokenStream() && _expectedTokens[id].type != CACHE_MISS)
	 {
	    **_config.getTokenStream() 
	       << _expectedTokens[id].timeSent.value()
	       << ";" << _expectedTokens[id].timeReceived.value()
	       << ";" << _expectedTokens[id].id
	       << ";" << _expectedTokens[id].source
	       << ";" << _expectedTokens[id].destination
	       << ";" << _expectedTokens[id].size
	       << std::endl;
	 }

	 if(_usePackets && _expectedTokens[id].type != CACHE_MISS) 
	 {
	    _config.addReceiveTime(_expectedTokens[id].destination);
	 }
	 
	 if(!_usePackets && _expectedTokens[id].type != CACHE_MISS)
	 {
	    _receivedTokens.push(_expectedTokens[id]);
	    _rxTokenEvent.notify(sc_core::SC_ZERO_TIME);
	 }
	 _expectedTokens.erase(_expectedTokens.find(id));
	 _incomingData.erase(_incomingData.find(id));	 
      }
    
      if(rxSpaceLeft() < 4 && _receivedTokens.empty())
      {
	 std::cout << "Receive buffer has no space left for a complete token: "
		   << rxSpaceLeft() << ";" << _measurements.rxUsed 
		   << std::endl;
	 /*
	
	   unsigned long int expecting = 0;
	   unsigned long int received = 0;
	   for(std::map<unsigned long int, tgToken>::iterator iter = 
	   _expectedTokens.begin(); iter != _expectedTokens.end(); 
	   ++iter)
	   {
	   std::cout << ":: ID " << iter->first << " expecting " 
	   << iter->second.size << " bytes (PORT:"
	   << iter->second.destination << ")" << std::endl;
	   expecting += iter->second.size;
	   }
	   for(std::map<unsigned long int, unsigned long int>::iterator iter = 
	   _incomingData.begin(); iter != _incomingData.end(); 
	   ++iter)
	   {
	   std::cout << ":: ID " << iter->first << " have received "
	   << iter->second << " bytes" << std::endl;
	   received += iter->second;
	   }
	   std::cout << ":: Expecting " << expecting 
	   << " bytes and have received " 
	   << received << " bytes" << std::endl;
	 */

	 //sc_core::sc_stop();
      }

      /*std::cout << "Packet received, rx buffer capacity " 
	<< _rxBufferLeft << " bytes" << std::endl;
      */
   }  

   unsigned long int Buffer::rxSpaceLeft()
   {
      return _rxBufferSize != 0 ? _rxBufferSize - _rxBufferUsed : UINT_MAX;
   }

   bool Buffer::rxTokenAvailable()
   {
      return !_receivedTokens.empty();
   }

   bool Buffer::rxPacketAvailable()
   {
      return !_receivedPackets.empty();
   }

   bool Buffer::rxCacheAvailable()
   {
      return !_receivedCaches.empty();
   }

   tgToken Buffer::rxGetToken()
   {
      if(_receivedTokens.empty())
      {
	 throw std::runtime_error("rxGetToken: no tokens to get");
      }
      tgToken token = _receivedTokens.front();
      _receivedTokens.pop();
      return token;
   }

   tgPacket* Buffer::rxGetPacket()
   {
//       std::cout << "CH RX packet taken "
// 		<< sc_core::sc_time_stamp().value()
// 		<< std::endl;


      if(_receivedPackets.empty())
      {
	 throw std::runtime_error("rxGetToken: no packets to get");
      }
      tgPacket* packet = _receivedPackets.front();
      _receivedPackets.pop();
      return packet;
   }

   tgPacket* Buffer::rxGetCache()
   {
      if(_receivedCaches.empty())
      {
	 throw std::runtime_error("rxGetCache: no cache packets to get");
      }
      tgPacket* packet = _receivedCaches.front();
      _receivedCaches.pop();
      return packet;
   }

   sc_core::sc_event* Buffer::rxGetReadEvent()
   {
      return &_rxReadEvent;
   }

   sc_core::sc_event* Buffer::rxGetTokenAvailableEvent()
   {
      return &_rxTokenEvent;
   }

   sc_core::sc_event* Buffer::rxGetPacketAvailableEvent()
   {
      _usePackets = true;
      return &_rxPacketEvent;
   }

   sc_core::sc_event* Buffer::rxGetCacheAvailableEvent()
   {
      return &_rxCacheEvent;
   }

   void Buffer::txPutPacket(tgPacket* packet)
   {
//       std::cout << "CH Got packet to send "
// 		<< sc_core::sc_time_stamp()
// 		<< std::endl;


      // Confirm that there is enough space in queue
      if(packet->size > txSpaceLeft())
      {
	 throw std::runtime_error
	    ("txPutPacket(): Packet doesn't fit to Buffer");
      }
      // Put packet to fifo, update space left counter and notify event
      _txQueue.push(packet);
      _txBufferUsed += packet->size;
      _txPacketEvent.notify(sc_core::SC_ZERO_TIME);
   }

   unsigned long int Buffer::txSpaceLeft()
   {
      return _txBufferSize != 0 ? _txBufferSize - _txBufferUsed : UINT_MAX;
   }

   bool Buffer::txPacketAvailable()
   {
      return !_txQueue.empty();
   }

   tgPacket* Buffer::txGetPacket()
   {
//       std::cout << "CH TX Packet taken "
// 		<< sc_core::sc_time_stamp().value()
// 		<< std::endl;
      
      // Check that fifo is not empty, update space left counter and notify
      // event
      tgPacket* retval;
      if(_txQueue.empty())
      {
	 throw std::runtime_error
	    ("txGetPacket(): Tried to read from empty buffer");
      }
      retval = _txQueue.front();
      _txQueue.pop();

      retval->data = new unsigned char[retval->size];

      unsigned int* ptr = (reinterpret_cast<unsigned int*>(retval->data));

      for(unsigned int i = 0; i < retval->size / 4; ++i)
      {
	 *ptr = retval->id;
	 ptr++;
      }

//       std::cout << "packet to " << std::hex << retval->address << std::dec
// 		<< " id  " << retval->id << std::endl;

      _txBufferUsed -= retval->size;
      _measurements.txBytes += retval->size;
      _txReadEvent.notify(sc_core::SC_ZERO_TIME);
      return retval;
   }

   sc_core::sc_event* Buffer::txGetPacketAvailableEvent()
   {
      return &_txPacketEvent;
   }

   sc_core::sc_event* Buffer::txGetReadEvent()
   {
      return &_txReadEvent;
   }

   void Buffer::expectToken(tgToken token)
   {
//         std::cout << "Expecting token to port " << token.destination 
//  		 << " with id " << token.id  
//  		 << " and size " << token.size << std::endl;
      _expectedTokens[token.id] = token;
   }

   void Buffer::removeToken(unsigned long int size)
   {    
      _rxBufferUsed -= size;
//       std::cout << "Token (" << size << " B) removed, rx buffer used " 
// 		<< _rxBufferUsed << " bytes" << std::endl;
      
      _rxReadEvent.notify(sc_core::SC_ZERO_TIME);
   }

   const BufferMeasurements& Buffer::getMeasurements()
   {
      _measurements.rxUsed = _rxBufferUsed;
      _measurements.txUsed = _txBufferUsed;
      return _measurements;
   }

   void Buffer::addInternalBytes(unsigned long int bytes)
   {
//       std::cout << "RX internal " << bytes << " bytes, used "
// 		<< _rxBufferUsed << std::endl;
      _measurements.internalBytes += bytes;
   }

   void Buffer::rxReserveSpace(unsigned long int bytes)
   {
//       std::cout << "RX reserve " << bytes << " bytes, used "
// 		<< _rxBufferUsed << std::endl;
      _rxBufferUsed += bytes;
   }

   void Buffer::rxFreeSpace(unsigned long int bytes)
   {
//       std::cout << "RX free " << bytes << " bytes, used "
// 		<< _rxBufferUsed << std::endl;
      _rxBufferUsed -= bytes;
   }

   void Buffer::txReserveSpace(unsigned long int bytes)
   {
      _txBufferUsed += bytes;
   }

   void Buffer::txFreeSpace(unsigned long int bytes)
   {
      _txBufferUsed -= bytes;
   }
 
   void Buffer::updateUnfinishedTokensLatency()
   {
      for(std::map<unsigned long int, tgToken>::iterator 
	     iter = _expectedTokens.begin(); 
	  iter != _expectedTokens.end(); ++iter)
      {
	 // Update receive time
	 sc_core::sc_time lat = sc_core::sc_time_stamp() - (*iter).second.timeSent;
	 _config.addTokenLatency((*iter).second.source, (*iter).second.destination, lat);
      }
   }
  
}

// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
