/**
 * @file amount.hh
 * @author Lasse Lehtonen
 * 
 * 
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: amount.hh 1916 2011-07-06 12:44:26Z lehton87 $
 *
 */

#ifndef SCTG_AMOUNT_HH
#define SCTG_AMOUNT_HH

#include "configuration.hh"

#include <boost/property_tree/ptree.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/foreach.hpp>
#include <boost/random.hpp>
#include <boost/optional.hpp>

#include <iostream>
#include <vector>
#include <exception>



namespace sctg
{
   /** Class for calculating polynoms and distributions
    *
    * Evaluates the polynom or distribution and returns a value
    */
   template <class Type>
   class Amount
   {
   public:    

      /** Constructor
       *
       *  @param[in] pt Property tree node containing polynomial or
       *  distributional tag
       *
       *  @param[in] config Reference to Configuration object
       */
      Amount(const boost::property_tree::ptree& pt,
	     Configuration& config);
    
      /** Destructor
       */
      virtual ~Amount();

      /** Evaluate distribution or polynom
       *
       * @return Result of distribution or polynom
       *
       * @param[in] x Variable for polynom or mean for distributions
       */
      Type value(Type x=0) const;    

    
    
   private:

      /** One polynomial term
       *
       */
      struct _Term
      {
	 Type coefficient; ///< Term's coefficient
	 Type exponent;    ///< Term's exponent
      };
    
      /** Polynom constructed from terms
       */
      std::vector<_Term> _polynom;

      /** Possible types objects can represent
       */
      enum _AmountType {POLYNOMIAL, UNIFORM, NORMAL};

      /** What this object represents
       */
      _AmountType _amountType;

      /** Pointer to uniform distribution object
       */
      boost::uniform_int<>* _uniform;
      /** Pointer to normal distribution object
       */
      boost::normal_distribution<double>* _normal;
      Type  _stdDev;

      /** Reference to random number generator for real numbers
       */
      Configuration::RealEngineType& _realEngine;
      /** Reference to random number generator for integers
       */
      Configuration::IntEngineType&  _intEngine;

   };
}


//
// CLASS IMPLEMENTATION
//



namespace sctg
{  
   template <class Type>
   Amount<Type>::Amount(const boost::property_tree::ptree& pt,
			Configuration& config)
      : _polynom(),
	_amountType(POLYNOMIAL),
	_uniform(0),            
	_normal(0),
	_stdDev(0),
	_realEngine(config.getRealRNGEngine()),
	_intEngine(config.getIntRNGEngine())
   {
      using boost::property_tree::ptree;
      ptree tempPTree;
    
      if(pt.find("polynomial") != pt.not_found())
      {
	 // Construct polynomial from property_tree
	 _amountType = POLYNOMIAL;	
	
	 // Store all terms
	 for(ptree::const_iterator iter = pt.get_child("polynomial").begin(); 
	     iter != pt.get_child("polynomial").end(); ++iter)
	 {      
	    // Ignore all but param tags
	    if((*iter).first == "param")
	    {		
	       _Term term = {
		  boost::lexical_cast<Type>
		  ((*iter).second.get<std::string>("<xmlattr>.value")),
		  boost::lexical_cast<Type>
		  ((*iter).second.get<std::string>("<xmlattr>.exp"))
	       };
		
	       _polynom.push_back(term);
	    }
	 }
      }
      else if(pt.find("distribution") != pt.not_found())
      {
	 // Create distribution
	 tempPTree = pt.get_child("distribution");
	 if(tempPTree.find("uniform") != tempPTree.not_found())
	 {
	    _amountType = UNIFORM;
	    Type min = pt.get<Type>("distribution.uniform.<xmlattr>.min");
	    Type max = boost::lexical_cast<Type>
	       (pt.get<std::string>("distribution.uniform.<xmlattr>.max"));
	    _uniform = new boost::uniform_int<>(min,max);
	 }
	 else if(tempPTree.find("normal") != tempPTree.not_found())
	 {
	    _amountType = NORMAL;
	    boost::optional<Type> mean = pt.get_optional<Type>
	       ("distribution.normal.<xmlattr>.mean");
	    _stdDev = pt.get<Type>
	       ("distribution.normal.<xmlattr>.standard_deviation");
	    if(mean)
	    {
	       _normal = new boost::normal_distribution<double>
		  (double(*mean), double(_stdDev));
	    }
	 }
	 else
	 {
	    throw std::runtime_error("Did not find valid distribution");
	 }      
      }    
      else
      {
	 throw std::runtime_error("No polynomial or distribution found");
      }      
   }
  
   //Destructor
   template <class Type>
   Amount<Type>::~Amount()
   {
      if(_normal)    
	 delete _normal;
      if(_uniform)
	 delete _uniform;
   }


   
   template <class Type>
   Type Amount<Type>::value(Type x) const
   {
      Type returnValue = 0;
      Type partialSum  = 0;

      switch(_amountType)
      {
	 case POLYNOMIAL:
	    // Loop over all terms and add them together
	    BOOST_FOREACH(_Term term, _polynom)
	    {
	       if(term.exponent == 0)
	       {
		  partialSum = 1;
	       }
	       else
	       {
		  // brute force power calculation
		  //  exponents whould be small so no need for optimization
		  partialSum = x;		
		  for(Type i = 1; i < term.exponent; ++i)
		  {
		     partialSum *= x;
		  }
	       }	    
	       returnValue += term.coefficient * partialSum;
	    }	
	    break;

	 case UNIFORM:
	    returnValue = (*_uniform)(_intEngine);
	    break;

	 case NORMAL:
	    if(_normal)
	    {
	       // Both mean and sigma were defined in xml
	       double result = _normal->operator () 
		  <Configuration::RealEngineType>(_realEngine);
	       returnValue = result < 1.0 ? 1 : Type(result);
	    }
	    else
	    {	   
	       // Take mean from parameter
	       double result =
		  boost::normal_distribution<>(x, _stdDev)(_realEngine);
	       returnValue = result < 1.0 ? 1 : Type(result);
	    }
	    break;
	
	 default:
	    throw;
	    break;
      }    

      return returnValue;
   }
      
} // End of namespace sctg

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
