/**
 *
 * @file packet_decoder_ctrl.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: packet_decoder_ctrl.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */



#ifndef ASEBT_SC_RTL_2_PKT_DECODER_HH
#define ASRBT_SC_RTL_2_PKT_DECODER_HH

#include <systemc>

namespace asebt
{
namespace sc_rtl_2
{
   
   template<int data_width_g    = 36,
	    int addr_width_g    = 32,
	    int pkt_len_g       = 0,
	    int fill_packet_g   = 0,
	    int len_flit_en_g   = 1,
	    int oaddr_flit_en_g = 1>
   class packet_decoder_ctrl : public sc_core::sc_module
   {
   public:
	
      sc_core::sc_in_clk                             clk;
      sc_core::sc_in<bool>              rst_n;
      sc_core::sc_in<sc_dt::sc_bv<data_width_g> >  net_data_in;
      sc_core::sc_in<bool>              net_empty_in;
      sc_core::sc_out<bool>             net_re_out;
      sc_core::sc_in<bool>              fifo_full_in;
      sc_core::sc_out<bool>             fifo_av_out;
      sc_core::sc_out<sc_dt::sc_bv<data_width_g> > fifo_data_out;
      sc_core::sc_out<bool>             fifo_we_out;
   
      SC_HAS_PROCESS(packet_decoder_ctrl);

      packet_decoder_ctrl(sc_core::sc_module_name name)
	 : sc_core::sc_module(name),
	   clk("clk"),
	   rst_n("rst_n"),
	   net_data_in("net_data_in"),
	   net_empty_in("net_empty_in"),
	   net_re_out("net_re_out"),
	   fifo_full_in("fifo_full_in"),
	   fifo_av_out("fifo_av_out"),
	   fifo_data_out("fifo_data_out"),
	   fifo_we_out("fifo_we_out"),
	   curr_state_r("curr_state_r"),
	   next_state("next_state"),
	   write_counter_r("write_counter_r"),
	   amount_r("amount_r"),
	   len_width_c(8)
      {
      
	 SC_METHOD(sync);
	 sensitive << clk.pos() << rst_n;

	 SC_METHOD(async);
	 sensitive << net_data_in << net_empty_in
		   << fifo_full_in << write_counter_r
		   << amount_r << curr_state_r;
      }
   
      virtual ~packet_decoder_ctrl()
      {
      
      }

      void sync()
      {
	 if(rst_n.read() == false)
	 {
	    curr_state_r    = START;
	    write_counter_r = 0;
	 }
	 else 
	 {
	    curr_state_r    = next_state;
	    amount_r        = 0;
	    write_counter_r = 0;
	 
	    switch(curr_state_r)
	    {
	       //
	       case READ_ADDRESS:
		  if(len_flit_en_g == 0 && 
		     net_empty_in.read() == false)
		  {
		     amount_r = net_data_in.read()
			.range(data_width_g-1, data_width_g - len_width_c)
			.to_ulong();
		  }
		  break;

	       case READ_AMOUNT:
		  if(len_flit_en_g == 1 && 
		     net_empty_in.read() == false)
		  {
		     if(data_width_g < 32)
		     {
			amount_r = net_data_in.read().to_uint() - oaddr_flit_en_g;
		     }
		     else
		     {
			amount_r = net_data_in.read().range(31, 0).to_uint() - 
			   oaddr_flit_en_g;
		     }
		  }
		  else
		  {
		     amount_r = amount_r;
		  }
		  break;

	       case READ_DST_ADDR:
		  amount_r = amount_r;
		  break;

	       case READ_DATA:
		  amount_r = amount_r;
		  if(fifo_full_in.read() == false &&
		     net_empty_in.read() == false)
		  {
		     if(write_counter_r == pkt_len_g)
		     {
			write_counter_r = write_counter_r;
		     }
		     else
		     {
			write_counter_r = write_counter_r + 1;
		     }
		  }
		  else
		  {
		     write_counter_r = write_counter_r;
		  }
		  break;
	       
	       default:
		  break;
	    }
	 }
      }

      void async()
      {
	 switch(curr_state_r)
	 {
	    case START:
	       net_re_out.write(false);
	       fifo_data_out.write("0");
	       fifo_av_out.write(false);
	       fifo_we_out.write(false);

	       if(net_empty_in.read() == false &&
		  fifo_full_in.read() == false)
	       {
		  next_state = READ_ADDRESS;
	       }
	       else
	       {
		  next_state = START;
	       }
	       break;

	    case READ_ADDRESS:
	       fifo_data_out
		  .write(net_data_in.read().range(data_width_g-len_width_c-1, 0));

	       if(oaddr_flit_en_g == 0)
	       {
		  fifo_av_out.write(true);
		  fifo_we_out.write(true);
	       }

	       if(fifo_full_in.read() == false &&
		  net_empty_in.read() == false)
	       {
		  next_state = READ_AMOUNT;
		  net_re_out.write(true);
	       }
	       else
	       {
		  next_state = READ_ADDRESS;
		  net_re_out.write(false);
		  fifo_we_out.write(false);
	       }
	       break;

	    case READ_AMOUNT:
	       fifo_data_out.write(net_data_in.read());
	       fifo_av_out.write(false);
	    
	       if(net_empty_in.read() == false &&
		  fifo_full_in.read() == false)
	       {
		  fifo_we_out.write(false);

		  if(len_flit_en_g == 1)
		  { net_re_out.write(true); }
		  else
		  { net_re_out.write(false); }
	       
		  if(oaddr_flit_en_g == 1)
		  { next_state = READ_DST_ADDR; }
		  else
		  { next_state = READ_DATA; }
	       }
	       else
	       {
		  next_state = READ_AMOUNT;
		  fifo_we_out.write(false);
		  net_re_out.write(false);
	       }
	       break;

	    case READ_DST_ADDR:
	       fifo_data_out.write(net_data_in.read());
	       fifo_av_out.write(true);

	       if(fifo_full_in.read() == false &&
		  net_empty_in.read() == false)
	       {
		  next_state = READ_DATA;
		  net_re_out.write(true);
		  fifo_we_out.write(true);
	       }
	       else
	       {
		  next_state = READ_DST_ADDR;
		  net_re_out.write(false);
		  fifo_we_out.write(false);
	       }
	       break;

	    case READ_DATA:
	       fifo_data_out.write(net_data_in.read());
	       fifo_av_out.write(false);

	       if(fifo_full_in.read() == false &&
		  net_empty_in.read() == false)
	       {
		  if((write_counter_r.read() == amount_r.read()-1 &&
		      fill_packet_g == 0) ||
		     (write_counter_r.read() == 
		      (pkt_len_g - 2 - len_flit_en_g - oaddr_flit_en_g) &&
		      fill_packet_g == 1))
		  {
		     next_state = START;
		  }
		  else
		  {
		     next_state = READ_DATA;
		  }

		  if(write_counter_r.read() < amount_r.read())
		  {
		     fifo_we_out.write(true);
		     net_re_out.write(true);
		  }
		  else
		  {
		     fifo_we_out.write(false);
		     net_re_out.write(true);
		  }	       
	       }
	       else
	       {
		  fifo_we_out.write(false);
		  net_re_out.write(false);
		  next_state = READ_DATA;
	       }
	       break;

	    default:
	       break;

	 }
      }
   
   
   private:

      enum states {START, READ_ADDRESS, READ_AMOUNT, READ_DST_ADDR, READ_DATA};
   
      sc_core::sc_signal<states> curr_state_r;
      sc_core::sc_signal<states> next_state;

      sc_core::sc_signal<int> write_counter_r;
      sc_core::sc_signal<int> amount_r;

      const unsigned int len_width_c;
   
   };

}
}

#endif



// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
