/* Copyright
 * ========================================================================================
 * Project:		Accurate DRAM Model
 * Author:		Nan Li, KTH
 * ID:			adm_controller.h, v1.1, 2010/11/27
 *
 * Description:	Defines the DRAM controller class
 *
 * ========================================================================================
 * Version History
 * ========================================================================================
 * Version 1.1: Tested, and fixed some bugs
 * Version 1.0: Added srmd extension support
 * Version 0.5: Added burst support for precise burst of type INCR
 * Version 0.2: Builded the basic structures, capable of single-data transfer
 * Version 0.1: Draft version
 * ========================================================================================
 */

#ifndef ADM_CONTROLLER_H_
#define ADM_CONTROLLER_H_

#include "systemc.h"
#include "ocpip.h"
#include "adm_dram_ctrl_ifs.h"
#include "adm_configuration.h"

#include <deque>

template <unsigned int BUSWIDTH = 32>
class adm_controller : public sc_module, public adm_ctrl_if
{
public:
	/* type definition */
	typedef typename ocpip::ocp_data_class_unsigned<BUSWIDTH, 32>::DataType Td;

	sc_in_clk clk;
	ocpip::ocp_slave_socket_tl1<BUSWIDTH> ocpPort;

	/* internal interface */
	sc_export<adm_ctrl_if> ctrlPort;
	sc_port<adm_dram_if> dramPort;

	/* Constructor */
	SC_HAS_PROCESS(adm_controller);
	adm_controller(const sc_module_name &nm, const adm_configuration &config)
			: sc_module(nm),
			  clk("clk"),
			  ocpPort("ocp_target_port"),
			  ctrlPort("ctrl_port"),
			  dramPort("dram_port"),
			  m_ocpReq(0),
			  m_ocpSrmdWriteReq(0),
			  m_ocpRsp(0),
			  m_burstReqBeatsRemain(0),
			  m_burstRspBeatsRemain(0),
			  m_dramRowSize(1 << config.m_columnAddressWidth),
			  m_dramWordsPerWrite(config.m_burstLength)
	{
		/* register transport function */
		ocpPort.register_nb_transport_fw(this, &adm_controller::nb_transport_fw);
		ocpPort.activate_synchronization_protection();

		/* register clocked process */
		SC_THREAD(proc);
		sensitive << clk.pos();
		dont_initialize();

		/* export the adm_ctrl_if interface */
		ctrlPort(*this);
	}

	/* Destructor */
	virtual ~adm_controller() {}

	/* Function for adm_ctrl_if */
	virtual bool putResponse(const adm_data &rsp) {
		m_admReadDataQueue.push_back(rsp);
		return true;
	}

	/* transport function in the forward path */
	tlm::tlm_sync_enum nb_transport_fw(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) {
		if (ph == tlm::BEGIN_REQ) {
			sc_assert(!m_ocpReq);
			m_ocpReq = &txn;
			m_dataHandshake = false;
			processRequest();
			if (!m_ocpReq) {
				ph = tlm::END_REQ;
				return tlm::TLM_UPDATED;
			}
		} else if (ph == ocpip::BEGIN_DATA) {
			sc_assert(!m_ocpReq);
			m_ocpReq = &txn;
			m_dataHandshake = true;
			processRequest();
			if (!m_ocpReq) {
				ph = ocpip::END_DATA;
				return tlm::TLM_UPDATED;
			}
		} else if (ph == tlm::END_RESP) {
			m_ocpRsp = 0;
		} else {
			sc_assert(false);
		}
		return tlm::TLM_ACCEPTED;
	}

private:
	/* clocked thread */
	void proc() {
		while (true) {
			/* wait for one delta cycle, to synchronize with DRAM core */
			wait(SC_ZERO_TIME);

			/* request path activities */
			sendRequest();
			tlm::tlm_generic_payload *txn = m_ocpReq;
			processRequest();
			if (txn != m_ocpReq) {
				/* transaction processed. send an END_REQ to the master if in
				 * REQ phase; send an END_DATA if in data handshake phase */
				tlm::tlm_phase phase;
				if (m_dataHandshake)
					phase = ocpip::END_DATA;
				else
					phase = tlm::END_REQ;

				sc_time time = SC_ZERO_TIME;
				ocpPort->nb_transport_bw(*txn, phase, time);
			}

			/* synchronize with the response of the DRAM core */
			wait(SC_ZERO_TIME);
			wait(SC_ZERO_TIME);

			/* response path activities */
			if (!m_ocpRsp) {
				processResponse();
				if (m_ocpRsp) {
					tlm::tlm_phase phase = tlm::BEGIN_RESP;
					sc_time time = SC_ZERO_TIME;
					tlm::tlm_sync_enum retVal = ocpPort->nb_transport_bw(*m_ocpRsp, phase, time);
					if ((retVal == tlm::TLM_UPDATED) && (phase == tlm::END_RESP)) {
						m_ocpRsp = 0;
					}
				}
			}

			/* wait for next clock edge */
			wait();
		} /* while (true) */
	}

	/* internal functions */
	/* process an incoming OCP request (or data handshake phase) into one or more (or none) internal
	 * memory access requests */
	void processRequest() {
		if (!m_ocpReq) return;
		if (!m_dataHandshake && !m_admRequestQueue.empty()) return;

		/* only precise burst is supported */
		sc_assert(!ocpip::extension_api::get_extension<ocpip::imprecise>(*m_ocpReq));

		/* only INCR burst is supported */
		ocpip::burst_sequence *bSeq;
		if (ocpip::extension_api::get_extension<ocpip::burst_sequence>(bSeq, *m_ocpReq))
			sc_assert(bSeq->value.sequence == ocpip::INCR);

		if (ocpip::extension_api::get_extension<ocpip::srmd>(*m_ocpReq)) {
			/* single request, multiple data - SRMD */

			if (m_ocpReq->get_command() == tlm::TLM_READ_COMMAND) {
				m_burstReqAddress = m_ocpReq->get_address() & (~(uint64)(sizeof(Td) - 1));
				m_burstReqBeatsRemain = calculateBurstLength(*m_ocpReq);

				m_admRequestQueue.push_back(adm_request());
				adm_request &req = m_admRequestQueue.back();
				req.address = m_burstReqAddress;
				req.command = tlm::TLM_READ_COMMAND;
				req.readLen = std::min(uint64(m_burstReqBeatsRemain * sizeof(Td)), m_dramRowSize - m_burstReqAddress % m_dramRowSize);

				m_burstReqBeatsRemain--;
				m_burstReqAddress += sizeof(Td);
				m_ocpTxnQueue.push_back(m_ocpReq);

				while (m_burstReqBeatsRemain) {
					if (m_burstReqAddress % m_dramRowSize == 0) {
						m_admRequestQueue.push_back(adm_request());
						adm_request &req = m_admRequestQueue.back();
						req.address = m_burstReqAddress;
						req.command = tlm::TLM_READ_COMMAND;
						req.readLen = std::min(m_burstReqBeatsRemain * sizeof(Td), m_dramRowSize);
					}
					m_burstReqBeatsRemain--;
					m_burstReqAddress += sizeof(Td);
					m_ocpTxnQueue.push_back(m_ocpReq);
				} /* while */

			} else if (m_ocpReq->get_command() == tlm::TLM_WRITE_COMMAND) {
				if (!m_burstReqBeatsRemain) {
					/* this is a write request (without data) */

					m_burstReqAddress = m_ocpReq->get_address() & (~(uint64)(sizeof(Td) - 1));
					m_burstWriteAddress = m_burstReqAddress;
					m_burstWriteDataOffset = 0;

					m_burstReqBeatsRemain = calculateBurstLength(*m_ocpReq);
				} else {
					/* this is an outstanding data phase */
					/* initiate an internal write at end of burst, or before internal write burst boundary */
					if ((m_burstReqBeatsRemain == 1) || ((m_burstReqAddress + sizeof(Td)) % (m_dramWordsPerWrite * sizeof(Td)) == 0)) {
						m_admRequestQueue.push_back(adm_request());
						adm_request &req = m_admRequestQueue.back();
						req.address = m_burstWriteAddress;
						req.command = tlm::TLM_WRITE_COMMAND;
						req.data.setLength(m_burstReqAddress - m_burstWriteAddress + sizeof(Td));
						req.data.copyFrom(m_ocpReq->get_data_ptr() + m_burstWriteDataOffset);

						m_burstWriteAddress = m_burstReqAddress + sizeof(Td);
						m_burstWriteDataOffset += req.data.getLength();

						/* give response to write when all data are collected */
						if (m_burstReqBeatsRemain == 1) {
							/* WRITE transaction has a response only when
							 * non-posted WRITE command (WRNP), or writeresp_enable
							 * configuration is set to 1. */
							if (!ocpip::extension_api::get_extension<ocpip::posted>(*m_ocpReq) ||
									ocpPort.get_resolved_ocp_config().writeresp_enable) {
								m_ocpSrmdWriteReq = m_ocpReq;
							}
						}
					}

					m_burstReqBeatsRemain--;
					m_burstReqAddress += sizeof(Td);
				}
			}

		} else {
			/* multiple request, multiple data - MRMD */

			if (m_ocpReq->get_command() == tlm::TLM_READ_COMMAND) {
				if (!m_burstReqBeatsRemain) {
					/* start a new burst */
					/* address is word-aligned */
					m_burstReqAddress = m_ocpReq->get_address() & (~(uint64)(sizeof(Td) - 1));

					m_burstReqBeatsRemain = calculateBurstLength(*m_ocpReq);

					m_admRequestQueue.push_back(adm_request());
					adm_request &req = m_admRequestQueue.back();
					req.address = m_burstReqAddress;
					req.command = tlm::TLM_READ_COMMAND;
					req.readLen = std::min(uint64(m_burstReqBeatsRemain * sizeof(Td)), m_dramRowSize - m_burstReqAddress % m_dramRowSize);
				} else {
					/* we are in the middle of a burst */
					/* only put a new internal request when starting a new row */
					if (m_burstReqAddress % m_dramRowSize == 0) {
						m_admRequestQueue.push_back(adm_request());
						adm_request &req = m_admRequestQueue.back();
						req.address = m_burstReqAddress;
						req.command = tlm::TLM_READ_COMMAND;
						req.readLen = std::min(m_burstReqBeatsRemain * sizeof(Td), m_dramRowSize);
					}
				}

				m_burstReqBeatsRemain--;
				m_burstReqAddress += sizeof(Td);

				m_ocpTxnQueue.push_back(m_ocpReq);

			} else if (m_ocpReq->get_command() == tlm::TLM_WRITE_COMMAND) {
				if (!m_burstReqBeatsRemain) {
					m_burstReqAddress = m_ocpReq->get_address() & (~(uint64)(sizeof(Td) - 1));
					m_burstWriteAddress = m_burstReqAddress;
					m_burstWriteDataOffset = 0;

					m_burstReqBeatsRemain = calculateBurstLength(*m_ocpReq);
				}

				/* initiate an internal write at end of burst, or before internal write burst boundary */
				if ((m_burstReqBeatsRemain == 1) || ((m_burstReqAddress + sizeof(Td)) % (m_dramWordsPerWrite * sizeof(Td)) == 0)) {
					m_admRequestQueue.push_back(adm_request());
					adm_request &req = m_admRequestQueue.back();
					req.address = m_burstWriteAddress;
					req.command = tlm::TLM_WRITE_COMMAND;
					req.data.setLength(m_burstReqAddress - m_burstWriteAddress + sizeof(Td));
					req.data.copyFrom(m_ocpReq->get_data_ptr() + m_burstWriteDataOffset);

					m_burstWriteAddress = m_burstReqAddress + sizeof(Td);
					m_burstWriteDataOffset += req.data.getLength();
				}

				m_burstReqBeatsRemain--;
				m_burstReqAddress += sizeof(Td);

				/* WRITE transaction has a response only when
				 * non-posted WRITE command (WRNP), or writeresp_enable
				 * configuration is set to 1. */
				if (!ocpip::extension_api::get_extension<ocpip::posted>(*m_ocpReq) ||
						ocpPort.get_resolved_ocp_config().writeresp_enable) {
					m_ocpTxnQueue.push_back(m_ocpReq);
				}
			} /* command */

		} /* srmd */

		/* clear the processed incoming request */
		m_ocpReq = 0;
	}

	/* send the internal request to memory */
	void sendRequest() {
		if (m_admRequestQueue.empty()) return;

		bool success = true;
		if (m_admRequestQueue.front().command == tlm::TLM_READ_COMMAND) {
			success = dramPort->putReadRequest(m_admRequestQueue.front().address, m_admRequestQueue.front().readLen);
		} else if (m_admRequestQueue.front().command == tlm::TLM_WRITE_COMMAND) {
			success = dramPort->putWriteRequest(m_admRequestQueue.front().address, m_admRequestQueue.front().data);
		}

		if (success) {
			m_admRequestQueue.pop_front();
		}

		if (m_ocpSrmdWriteReq && m_admRequestQueue.empty()) {
			m_ocpTxnQueue.push_back(m_ocpSrmdWriteReq);
			m_ocpSrmdWriteReq = 0;
		}
	}

	/* copy the data of the internal response into the ocp transaction. */
	void processResponse() {
		if (m_ocpRsp) return;
		if (m_ocpTxnQueue.empty()) return;
		if (m_admReadDataQueue.empty() &&
				(m_ocpTxnQueue.front()->get_command() == tlm::TLM_READ_COMMAND) &&
				(!m_burstRspBeatsRemain || (m_burstReadDataOffset == m_burstReadDataOffsetNext)))
			return;

		m_ocpRsp = m_ocpTxnQueue.front();
		m_ocpTxnQueue.pop_front();

		/* only read command needs to deal with data in response */
		if (m_ocpRsp->get_command() == tlm::TLM_READ_COMMAND) {
			if (!m_burstRspBeatsRemain) {
				/* a new response burst */
				m_burstRspBeatsRemain = calculateBurstLength(*m_ocpRsp);
				m_burstReadDataOffset = 0;

				m_admReadDataQueue.front().copyTo(m_ocpRsp->get_data_ptr());
				m_burstReadDataOffsetNext = m_admReadDataQueue.front().getLength();
				m_admReadDataQueue.pop_front();
			} else {
				/* in the middle of a response burst */
				/* copy data from internal response only when data offset reaches next value */
				if (m_burstReadDataOffset == m_burstReadDataOffsetNext) {
					m_admReadDataQueue.front().copyTo(m_ocpRsp->get_data_ptr() + m_burstReadDataOffset);
					m_burstReadDataOffsetNext += m_admReadDataQueue.front().getLength();
					m_admReadDataQueue.pop_front();
				}
			}
			m_burstRspBeatsRemain--;
			m_burstReadDataOffset += sizeof(Td);
		}

		m_ocpRsp->set_response_status(tlm::TLM_OK_RESPONSE);
	}

	unsigned int calculateBurstLength(tlm::tlm_generic_payload &txn) {
		ocpip::burst_length *bLen;
		if (ocpip::extension_api::get_extension<ocpip::burst_length>(bLen, txn))
			return bLen->value;
		else {
			SC_REPORT_WARNING(SC_ID_WITHOUT_MESSAGE_, "burst_length extension is not used. Calculating burst length from data length.");
			return txn.get_data_length() / sizeof(Td);
		}
	}

private:
	/* internal type */
	struct adm_request {
		tlm::tlm_command command;
		uint64 address;
		adm_data data;
		int readLen;
	};

	/* internal state variables */
	/* forward path (request) */
	tlm::tlm_generic_payload *m_ocpReq;
	tlm::tlm_generic_payload *m_ocpSrmdWriteReq;
	std::deque<adm_request> m_admRequestQueue;
	bool m_dataHandshake;

	/* transaction queue for the pipeline */
	std::deque<tlm::tlm_generic_payload *> m_ocpTxnQueue;

	/* backward path (response) */
	tlm::tlm_generic_payload * m_ocpRsp;
	std::deque<adm_data> m_admReadDataQueue;

	/* burst tracker */
	/* request burst */
	uint64 m_burstReqAddress;	/* current address of a burst */
	unsigned int m_burstReqBeatsRemain; /* number of remaining beats */
	uint64 m_burstWriteAddress;	/* address of next internal write */
	int m_burstWriteDataOffset; /* offset in the transaction's data array */

	/* response burst */
	unsigned int m_burstRspBeatsRemain; /* number of remaining beats in the response */
	int m_burstReadDataOffset; /* offset in the response transaction's data array */
	int m_burstReadDataOffsetNext; /* next offset that needs a data copy from internal response */

	/* DRAM parameters that are used in this module */
	unsigned int m_dramRowSize;
	unsigned int m_dramWordsPerWrite;
};

#endif /* ADM_CONTROLLER_H_ */
