// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary


#ifndef _SlaveThreadedRespQ_h
#define _SlaveThreadedRespQ_h

#include "SlaveResponseQueue.h"

// model a collection of response queues - one for each thread
template <typename TdataCl>
class ThreadedRespQ {

  public:
    // type definitions
    typedef typename TdataCl::DataType Td;
    typedef typename TdataCl::AddrType Ta;

  private:
    int m_threads;              // Number of threads & internal queues
    ResponseQueue<TdataCl> *respQ; // Array of internal queues - one for each thread
    bool m_WaitOnNewResp;
    sc_event m_NewQRespEvent;
    int m_lastThreadSent;

  public:
    ThreadedRespQ(int numThreads)
      : m_threads(numThreads),
        m_WaitOnNewResp(false),
        m_lastThreadSent(0)
    {
        // Create my queues
        assert(numThreads > 0);
        respQ = new ResponseQueue<TdataCl>[numThreads];
        reset();
    }
    
    ~ThreadedRespQ()
    {
        delete[] respQ;
    }

    void reset()
    {
        for (int i = 0; i < m_threads; i++) {
            respQ[i].reset();
        }

        // TODO: Are these needed?
        m_WaitOnNewResp = false;
    }

    void enqueueBlocking(const OCPResponseGrp<Td>& resp, unsigned int chunklen, bool chunklast, const sc_time& send_time)
    {
        // check for full
        int myThread = resp.SThreadID;
        assert(myThread >= 0);
        assert(myThread < m_threads);
        respQ[myThread].enqueueBlocking(resp, chunklen, chunklast, send_time);
        m_NewQRespEvent.notify();
    }

    // Is there a response ready to go out?
    bool respReady()
    {
        purgePlaceholders();
        for (int i=0; i<m_threads; i++) {
            if (!respQ[i].empty()) {
                // At least one not empty
                return true;
            }
        }
        // All empty
        return false;
    }

    // dequeueBlocking uses the following algorithm: 
    // Purge all placeholder items
    // Find the response with the earliest send time.
    // If more than one response has the same earliest time, go round-robin through them.
    // return the response with the earliest send time time.
    // if no responese are available, wait for a new response to come in and try again.
    void dequeueBlocking(OCPResponseGrp<Td>& resp, unsigned int& chunklen, bool& chunklast, sc_time& send_time)
    {
        while (!respReady()) {
            // No respone to send.
            if (!m_WaitOnNewResp) {
                // go to the dequeue waiting state

                m_WaitOnNewResp = true;
                wait(m_NewQRespEvent);

                m_WaitOnNewResp = false;
            } else {
                cerr << "ERROR: Cannot handle two processes waiting to dequeue from SlaveThreadedRespQ."
                     << endl;
                sc_stop();
            }
        }

        // There is a response waiting to be sent

        // Find earliest send time; 
        bool respFound = false;
        sc_time thisQTime = SC_ZERO_TIME;
        sc_time earliestTime = SC_ZERO_TIME;
        int earliestThread = -1;
        int thisThread;

        for (int i=1; i<=m_threads; i++) {
            thisThread = (i+m_lastThreadSent)%(m_threads);
            if ( !respQ[thisThread].empty() ) {
                respQ[thisThread].getHeadSendTime(thisQTime);
                if ( (!respFound) || (thisQTime < earliestTime) ) {
                    respFound = true;
                    earliestTime = thisQTime;
                    earliestThread = thisThread;
                }
            }
        }

        assert(respFound == true);
        
        unsigned int chunklen_v;
        bool chunklast_v;

        respQ[earliestThread].dequeueBlocking(resp, chunklen_v, chunklast_v, send_time);
        m_lastThreadSent = earliestThread;

        chunklen = chunklen_v;
        chunklast = chunklast_v;
    }

    // True if all response queues are empty
    bool empty()
    {
        for (int i=0; i<m_threads; i++) {
            if (!respQ[i].empty()) {
                return false;
            }
        }
        return true;
    }

    bool getHeadSResp(OCPSRespType& headSResp, int threadNum)
    {
        return respQ[threadNum].getHeadSResp(headSResp);
    }

    bool getHeadSendTime(sc_time& headSendTime, int threadNum)
    {
        return respQ[threadNum].getHeadSendTime(headSendTime);
    }

    int length(int threadNum)
    {
        respQ[threadNum].purgePlaceholders();
        return respQ[threadNum].length();
    }

    void purgePlaceholders()
    {
        // Purge all placeholders
        for (int i=0; i<m_threads; i++) {
            respQ[i].purgePlaceholders();
        }
    }

    ResponseQueue<TdataCl>* GetRespQ() {
        return respQ;
    }


};

#endif // _SlaveThreadedRespQ_h
