// 
//  Copyright 2003, 2004 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//         $Id: ocp_tl1_master_sync.cpp,v 1.2 2004/09/05 17:57:56 halexan Exp $
//
//  Description : Transaction Level - Layer-1 example Master
//    Features:
//    - Synchronous master using OCP data class
//    - Non-blocking methods are used
//    - Pipelined Request and Response threads
//    - Two-cycle response.
//                
// Parameter    :
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - DelayPct: Non-negative number specifying the delay in percent
//                 of a clock cycle, the data are transmitted.
// ============================================================================

#include "ocp_tl1_master_sync.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Sync::OCP_TL1_Master_Sync
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Sync<TdataCl>::OCP_TL1_Master_Sync
      (
        sc_module_name name_
      , int ID
      , int Priority
      , double DelayPct
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_DelayPct        (DelayPct)
{
  // InitParameters();

  SC_THREAD(MasterRequest);
  sensitive_pos(clk); 
  SC_THREAD(MasterResponse);
  sensitive_pos(clk); 
}

// ----------------------------------------------------------------------------
// Process : OCP_TL1_Master_Sync::~OCP_TL1_Master_Sync
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL1_Master_Sync<TdataCl>::~OCP_TL1_Master_Sync()
{
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Sync::MasterRequest()
//
//  Top level request method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Sync<TdataCl>::MasterRequest()
{
  int i = 0;
  int Pause[] = {0, 0, 0, 0}; // # of cycles between requests;
//  int Pause[] = {0, 1, 2, 3}; // # of cycles between requests;
  bool tmp = 0;
  Ta Addr = 0;
  int cnt = 0;

  while(true)
  {
    wait(); // wait for rising edge of clock

    if (cnt == 0)
    {
      // check if channel is free
      tmp = !MasterP->MgetSbusy();
      if (tmp)
      {
        // get new Pause value
        cnt = Pause[i%4]; i++;

        // Delay request for simulating physical signal delay
	// This must work with or without the next line
        // The delay must be less than 1 clk cycle
	//	if (m_DelayPct > 0)
        //  wait(TL_CLK_PERIOD * (m_DelayPct/100.0), TL_TIME_UNIT);

        // Put request into the channel
	m_DataCl->MputMAddr(Addr);
        MasterP->MputReadRequest(); 

#ifdef DEBUG_G1
      cout << "Master sent request "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " address " << Addr << endl;
#endif

        Addr += 4;
      }
    }
    else
    {
      cnt--;
    }
  } // end of while loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Sync::MasterResponse()
//
//  Top level response method of the Master
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Sync<TdataCl>::MasterResponse()
{
  Td Rdata;


  while(true)
  {
    wait(); // for rising clock edge
    if (MasterP->MgetResponse(true))
    {
      m_DataCl->MgetSData(Rdata);

#ifdef DEBUG_G1
      cout << "Master got response "
           << " delta " << simcontext()->delta_count()
           << " time " << sc_time_stamp().to_seconds()
           << " data " << Rdata << endl;
#endif
    }
  } // end of while(true) loop
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Sync::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL1_Master_Sync<TdataCl>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Master_Sync::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL1_Master_Sync<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get communication structure
  m_CommCl = MasterP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into Channel
  m_ParamCl->respaccept = true;

}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL1_Master_Sync<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
