///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  TL1/TL0 OCP Master Adapter
//
// SystemC Module Adapting between a TLM2-based OCP-TL1 master interface
// and a TL0 (signal-level) OCP interface
// Typical use is between a TL1 OCP initiator and a TL0 OCP target
//
// Module is a class template.  Template parameters are the TLM2 bus width
// and a traits class defining the types of the TL0 signals
//
// Timing:
//   Module is insensitive to timing on TL1 side
//   User needs to tell Module when to sample on TL0 side
//   Module will publish its TL1 timing using normal Tl1 mechanism
//   Module's timing on TL0 side depends on user-supplied sample times
//   and TL1 timing and can be obtained using the show_TL0_timing(x) API.
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef OCPIP_VERSION
  #error tl1_initiator_to_tl0_target.h may not be included directly. Use #include "ocpip_adapters.h" or #include "ocpip_adapters_X_X_X.h" (where desired ocp version is X.X.X)
#endif


// namespace
namespace OCPIP_VERSION { namespace tl1_tl0 {


// non-template base class containing actual functionality
// implementation in cpp file
class tl1_to_tl0_base: public sc_core::sc_module {
public:
  virtual void end_of_elaboration();
  typedef void (tl1_to_tl0_base::*callable)();

protected:
  sc_core::sc_in<bool> clk_port;

  // constructor/destructor only callable from the derived adapter template
  tl1_to_tl0_base(sc_core::sc_module_name n);
  virtual ~tl1_to_tl0_base();

  // methods for communicating with the derived adapter template
  //   functions called by this base class are pure virtual
  //   functions called by the derived adapter template are implemented

  // TL1 TLM2 transport
  virtual tlm::tlm_sync_enum nb_transport_bw(
    tlm::tlm_generic_payload &pl, tlm::tlm_phase &ph, sc_core::sc_time &ta) = 0;
  tlm::tlm_sync_enum nb_transport_fw(
    tlm::tlm_generic_payload &pl, tlm::tlm_phase &ph, sc_core::sc_time &ta);

  // TL0 port access; getters and setters

  // clk-enable
  virtual bool enableclk() = 0;

  // basic
  virtual void maddr(sc_dt::uint64) = 0;
  virtual void mcmd(unsigned) = 0;  unsigned new_mcmd;  bool mcmd_wr;
  virtual void mdata(unsigned char, unsigned) = 0;
  virtual void mdata() = 0;
  virtual void mdatavalid(bool) = 0;  bool new_mdatavalid;  bool mdatavalid_wr;
  virtual void mrespaccept(bool) = 0;  bool new_mrespaccept;  bool mrespaccept_wr;
  virtual unsigned char sdata(unsigned) = 0;
  virtual bool scmdaccept() = 0;
  virtual bool sdataaccept() = 0;
  virtual unsigned sresp() = 0;

  // simple
  virtual void maddrspace(unsigned) = 0;
  virtual void mbyteen(sc_dt::uint64) = 0;
  virtual void mdatabyteen(sc_dt::uint64) = 0;
  virtual void mdatainfo(tlm::tlm_generic_payload *, unsigned) = 0;
  virtual void mreqinfo(tlm::tlm_generic_payload *) = 0;
  virtual void sdatainfo(tlm::tlm_generic_payload *, unsigned) = 0;
  virtual void srespinfo(tlm::tlm_generic_payload *) = 0;

  // burst
  virtual void matomiclength(unsigned) = 0;
  virtual void mblockheight(unsigned) = 0;
  virtual void mblockstride(sc_dt::uint64) = 0;
  virtual void mburstlength(unsigned) = 0;
  virtual void mburstprecise(bool) = 0;
  virtual void mburstseq(unsigned) = 0;
  virtual void mburstsinglereq(bool) = 0;
  virtual void mdatalast(bool) = 0;
  virtual void mdatarowlast(bool) = 0;
  virtual void mreqlast(bool) = 0;
  virtual void mreqrowlast(bool) = 0;
  virtual bool sresplast() = 0;
  virtual bool sresprowlast() = 0;

  // tag
  virtual void mdatatagid(unsigned) = 0;
  virtual void mtagid(unsigned) = 0;
  virtual void mtaginorder(bool) = 0;
  virtual unsigned stagid() = 0;
  virtual bool staginorder() = 0;

  // thread
  virtual void mconnid(unsigned) = 0;
  virtual void mdatathreadid(unsigned) = 0;
  virtual void mthreadbusy(unsigned) = 0;
  virtual void mthreadid(unsigned) = 0;
  virtual unsigned sdatathreadbusy() = 0;
  virtual unsigned sthreadbusy() = 0;
  virtual unsigned sthreadid() = 0;

  // sideband
  virtual void merror(bool) = 0;
  virtual void mflag(tlm::tlm_generic_payload *) = 0;
  virtual void mreset_n(bool) = 0;
  virtual bool serror() = 0;
  virtual const sc_core::sc_event &serror_event() = 0;
  virtual void sflag(tlm::tlm_generic_payload *) = 0;
  virtual const sc_core::sc_event &sflag_event() = 0;
  virtual bool sinterrupt() = 0;
  virtual const sc_core::sc_event &sinterrupt_event() = 0;
  virtual bool sreset_n() = 0;
  virtual const sc_core::sc_event &sreset_event() = 0;

  // transaction objects for thread-busy signalling
  virtual tlm::tlm_generic_payload *get_tb_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_flag_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_error_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_interrupt_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_reset_transaction() = 0;
  virtual void ocp_socket_reset() = 0;

  ocp_parameters ocp_config;
  unsigned byte_wdth;

  sc_core::sc_time response_t;
  sc_core::sc_time accept_t;
  sc_core::sc_time data_accept_t;
  sc_core::sc_time threadbusy_t;
  sc_core::sc_time data_threadbusy_t;
  sc_core::sc_time cmd_t;
  sc_core::sc_time data_t;

private:
  tlm::tlm_generic_payload *oldest_txn, *newest_txn;

  sc_core::sc_event response_e;
  sc_core::sc_event accept_e;
  sc_core::sc_event data_accept_e;
  sc_core::sc_event threadbusy_e;
  sc_core::sc_event data_threadbusy_e;

  void clk_rising_edge();
  void response_sample();
  void accept_sample();
  void data_accept_sample();
  void threadbusy_sample();
  void data_threadbusy_sample();
  void handle_sreset();
  void handle_sflag();
  void handle_sinterrupt();
  void handle_serror();

  // specifically NON-delta-protected variables for inter-process communication
  bool cmdaccept_state, dataaccept_state;
  tlm::tlm_generic_payload *curr_tl0_cmd, *curr_tl0_data, *curr_tl0_resp;
  unsigned curr_sthreadbusy, curr_sdatathreadbusy;

  bool curr_serror, curr_sinterrupt;
  unsigned curr_sflag;

  // lower-level functionality
  void end_resp(tlm::tlm_generic_payload &pl);
  void remove_from_list(tlm::tlm_generic_payload &pl);
  void add_transaction(tlm::tlm_generic_payload &pl);
  void exit_reset();

  // process for writing to critical signals from more than one place
  void writer_process();
  sc_core::sc_event writer_trigger;
};


// end tl1_tl0 namespace
}


// template for user-instantiable adapter module
template<
  int BUSWIDTH,
  class TL0_TYPES = tl1_tl0::POD_DEFAULT_TL0_TYPES<>,
  int MRESET_N_INITIAL = 1
>
class tl1_initiator_to_tl0_target: public tl1_tl0::tl1_to_tl0_base {
typedef tl1_tl0::tl1_to_tl0_base base;
public:
  // constructor
  tl1_initiator_to_tl0_target(
      sc_core::sc_module_name n, tl1_tl0::get_TL0_timing_base *gtb = 0):
    base(n),
    Clk(clk_port),
    ocpTL1("OCP_TL1", this, &tl1_initiator_to_tl0_target::master_timing_update),
    EnableClk("EnableClk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    MDataValid("MDataValid"),
    MRespAccept("MRespAccept"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SDataAccept("SDataAccept"),
    SResp("SResp"),
    MAddrSpace("MAddrSpace"),
    MByteEn("MByteEn"),
    MDataByteEn("MDataByteEn"),
    MDataInfo("MDataInfo"),
    MReqInfo("MReqInfo"),
    SDataInfo("SDataInfo"),
    SRespInfo("SRespInfo"),
    MAtomicLength("MAtomicLength"),
    MBlockHeight("MBlockHeight"),
    MBlockStride("MBlockStride"),
    MBurstLength("MBurstLength"),
    MBurstPrecise("MBurstPrecise"),
    MBurstSeq("MBurstSeq"),
    MBurstSingleReq("MBurstSingleReq"),
    MDataLast("MDataLast"),
    MDataRowLast("MDataRowLast"),
    MReqLast("MReqLast"),
    MReqRowLast("MReqRowLast"),
    SRespLast("SRespLast"),
    SRespRowLast("SRespRowLast"),
    MDataTagID("MDataTagID"),
    MTagID("MTagID"),
    MTagInOrder("MTagInOrder"),
    STagID("STagID"),
    STagInOrder("STagInOrder"),
    MConnID("MConnID"),
    MDataThreadID("MDataThreadID"),
    MThreadBusy("MThreadBusy"),
    MThreadID("MThreadID"),
    SDataThreadBusy("SDataThreadBusy"),
    SThreadBusy("SThreadBusy"),
    SThreadID("SThreadID"),
    MError("MError"),
    MFlag("MFlag"),
    MReset_n("MReset_n"),
    SError("SError"),
    SFlag("SFlag"),
    SInterrupt("SInterrupt"),
    SReset_n("SReset_n"),
    eoe_called(false),
    timing_received(false),
    get_TL0_timing(gtb)
  {
    // forward TLM2 transport calls go into the base class
    ocpTL1.register_nb_transport_fw(this, &base::nb_transport_fw);
    ocpTL1.activate_synchronization_protection();

    MCmd.initialize(typename TL0_TYPES::MCMD_T(tl1_tl0::CMD_IDLE));
    MDataValid.initialize(typename TL0_TYPES::MDATAVALID_T(false));
    MRespAccept.initialize(typename TL0_TYPES::MRESPACCEPT_T(false));
    MReset_n.initialize(typename TL0_TYPES::MRESET_N_T(MRESET_N_INITIAL));
  }

  // setup functions called by instantiator eg wrapper around TL0 module

  void set_sample_times(
    const sc_core::sc_time &resp,
    const sc_core::sc_time &accept = sc_core::SC_ZERO_TIME,
    const sc_core::sc_time &data_accept = sc_core::SC_ZERO_TIME,
    const sc_core::sc_time &threadbusy = sc_core::SC_ZERO_TIME,
    const sc_core::sc_time &data_threadbusy = sc_core::SC_ZERO_TIME)
  {
    // set sample times in base class - inputs equal to 0 are ignored
    if(resp > sc_core::SC_ZERO_TIME) base::response_t = resp;
    if(accept > sc_core::SC_ZERO_TIME) base::accept_t = accept;
    if(data_accept > sc_core::SC_ZERO_TIME) base::data_accept_t = data_accept;
    if(threadbusy > sc_core::SC_ZERO_TIME) base::threadbusy_t = threadbusy;
    if(data_threadbusy > sc_core::SC_ZERO_TIME) base::data_threadbusy_t = data_threadbusy;
  }

  void show_TL0_timing(ocp_tl1_master_timing times) {
    if(get_TL0_timing != 0) {
      sc_core::sc_time unit = sc_core::sc_get_time_resolution();
      times.RequestGrpStartTime += unit;
      times.DataHSGrpStartTime += unit;
      times.MRespAcceptStartTime += unit;
      times.MThreadBusyStartTime += unit;
      get_TL0_timing->master(this, times);
    }
  }

  sc_core::sc_time calc_cmdacc_t() {
    return sc_dt::sc_max(
      base::cmd_t + sc_core::sc_get_time_resolution(),
      base::accept_t);
  }

  sc_core::sc_time calc_dataacc_t() {
    return sc_dt::sc_max(
      base::data_t + sc_core::sc_get_time_resolution(),
      base::data_accept_t);
  }

  void master_timing_update(ocp_tl1_master_timing times) {
    // called from socket to provide TL1 timing of connected master
    show_TL0_timing(times);
    timing_received = true;
    // Cmd/Data start times may affect Cmd/DataAcceptStartTime
    sc_core::sc_time old_tl1_cmdaccept_t = calc_cmdacc_t();
    base::cmd_t = times.RequestGrpStartTime;
    sc_core::sc_time new_tl1_cmdaccept_t = calc_cmdacc_t();
    sc_core::sc_time old_tl1_dataaccept_t = calc_dataacc_t();
    base::data_t = times.DataHSGrpStartTime;
    sc_core::sc_time new_tl1_dataaccept_t = calc_dataacc_t();
    if(((old_tl1_cmdaccept_t != new_tl1_cmdaccept_t) ||
        (old_tl1_dataaccept_t != new_tl1_dataaccept_t)) && eoe_called)
      set_slave_timing();
  }

  void end_of_elaboration() {
    base::end_of_elaboration();
    eoe_called = true;
    set_slave_timing();
    if(!timing_received) show_TL0_timing(ocp_tl1_master_timing());
  }

  void set_slave_timing() {
    // publish timing on TL1: the adapter does not have "default timing"
    ocp_tl1_slave_timing times;
    times.SCmdAcceptStartTime = calc_cmdacc_t();
    times.SDataAcceptStartTime = calc_dataacc_t();
    times.ResponseGrpStartTime = base::response_t;
    times.SThreadBusyStartTime = base::threadbusy_t;
    times.SDataThreadBusyStartTime = base::data_threadbusy_t;
    ocpTL1.set_slave_timing(times);
  }

  void set_ocp_config(const ocp_parameters &P) {
    // OCP socket is not generic because attached to TL0 OCP interface which
    //   must have a known configuration
    ocpTL1.set_ocp_config(P);
    base::ocp_config = P;
  }

  // SystemC communications
  // clock (reference to actual clock object in base class)
  sc_core::sc_in<bool> &Clk;

  // TL1 socket
  ocp_slave_socket_tl1<BUSWIDTH> ocpTL1;

  // TL0 ports - those whose type is TIEOFF<> effectively do not exist
  // clk-enable
  sc_core::sc_in<typename TL0_TYPES::ENABLE_CLK_T> EnableClk;

  // basic
  sc_core::sc_out<typename TL0_TYPES::MADDR_T> MAddr;
  sc_core::sc_out<typename TL0_TYPES::MCMD_T> MCmd;
  sc_core::sc_out<typename TL0_TYPES::DATA_T> MData;
  sc_core::sc_out<typename TL0_TYPES::MDATAVALID_T> MDataValid;
  sc_core::sc_out<typename TL0_TYPES::MRESPACCEPT_T> MRespAccept;
  sc_core::sc_in<typename TL0_TYPES::DATA_T> SData;
  sc_core::sc_in<typename TL0_TYPES::SCMDACCEPT_T> SCmdAccept;
  sc_core::sc_in<typename TL0_TYPES::SDATAACCEPT_T> SDataAccept;
  sc_core::sc_in<typename TL0_TYPES::SRESP_T> SResp;

  // simple
  sc_core::sc_out<typename TL0_TYPES::MADDRSPACE_T> MAddrSpace;
  sc_core::sc_out<typename TL0_TYPES::MBYTEEN_T> MByteEn;
  sc_core::sc_out<typename TL0_TYPES::MDATABYTEEN_T> MDataByteEn;
  sc_core::sc_out<typename TL0_TYPES::MDATAINFO_T> MDataInfo;
  sc_core::sc_out<typename TL0_TYPES::MREQINFO_T> MReqInfo;
  sc_core::sc_in<typename TL0_TYPES::SDATAINFO_T> SDataInfo;
  sc_core::sc_in<typename TL0_TYPES::SRESPINFO_T> SRespInfo;

  // burst
  sc_core::sc_out<typename TL0_TYPES::MATOMICLENGTH_T> MAtomicLength;
  sc_core::sc_out<typename TL0_TYPES::MBLOCKHEIGHT_T> MBlockHeight;
  sc_core::sc_out<typename TL0_TYPES::MBLOCKSTRIDE_T> MBlockStride;
  sc_core::sc_out<typename TL0_TYPES::MBURSTLENGTH_T> MBurstLength;
  sc_core::sc_out<typename TL0_TYPES::MBURSTPRECISE_T> MBurstPrecise;
  sc_core::sc_out<typename TL0_TYPES::MBURSTSEQ_T> MBurstSeq;
  sc_core::sc_out<typename TL0_TYPES::MBURSTSINGLEREQ_T> MBurstSingleReq;
  sc_core::sc_out<typename TL0_TYPES::MDATALAST_T> MDataLast;
  sc_core::sc_out<typename TL0_TYPES::MDATAROWLAST_T> MDataRowLast;
  sc_core::sc_out<typename TL0_TYPES::MREQLAST_T> MReqLast;
  sc_core::sc_out<typename TL0_TYPES::MREQROWLAST_T> MReqRowLast;
  sc_core::sc_in<typename TL0_TYPES::SRESPLAST_T> SRespLast;
  sc_core::sc_in<typename TL0_TYPES::SRESPROWLAST_T> SRespRowLast;

  // tag
  sc_core::sc_out<typename TL0_TYPES::MDATATAGID_T> MDataTagID;
  sc_core::sc_out<typename TL0_TYPES::MTAGID_T> MTagID;
  sc_core::sc_out<typename TL0_TYPES::MTAGINORDER_T> MTagInOrder;
  sc_core::sc_in<typename TL0_TYPES::STAGID_T> STagID;
  sc_core::sc_in<typename TL0_TYPES::STAGINORDER_T> STagInOrder;

  // thread
  sc_core::sc_out<typename TL0_TYPES::MCONNID_T> MConnID;
  sc_core::sc_out<typename TL0_TYPES::MDATATHREADID_T> MDataThreadID;
  sc_core::sc_out<typename TL0_TYPES::MTHREADBUSY_T> MThreadBusy;
  sc_core::sc_out<typename TL0_TYPES::MTHREADID_T> MThreadID;
  sc_core::sc_in<typename TL0_TYPES::SDATATHREADBUSY_T> SDataThreadBusy;
  sc_core::sc_in<typename TL0_TYPES::STHREADBUSY_T> SThreadBusy;
  sc_core::sc_in<typename TL0_TYPES::STHREADID_T> SThreadID;

  // sideband
  sc_core::sc_out<typename TL0_TYPES::MERROR_T> MError;
  sc_core::sc_out<typename TL0_TYPES::MFLAG_T> MFlag;
  sc_core::sc_out<typename TL0_TYPES::MRESET_N_T> MReset_n;
  sc_core::sc_in<typename TL0_TYPES::SERROR_T> SError;
  sc_core::sc_in<typename TL0_TYPES::SFLAG_T> SFlag;
  sc_core::sc_in<typename TL0_TYPES::SINTERRUPT_T> SInterrupt;
  sc_core::sc_in<typename TL0_TYPES::SRESET_N_T> SReset_n;

private:
  // calls from base class

  // backward TLM2 transport calls go to the OCP socket
  tlm::tlm_sync_enum nb_transport_bw(
    tlm::tlm_generic_payload &pl, tlm::tlm_phase &ph, sc_core::sc_time &ta) {
    return ocpTL1->nb_transport_bw(pl, ph, ta);
  }

  // TL0 port access

  // clk-enable
  bool enableclk() {return tl1_tl0::port_to_bool(EnableClk);}

  // basic
  void maddr(sc_dt::uint64 d) {MAddr = typename TL0_TYPES::MADDR_T(d);}
  void mcmd(unsigned d) {MCmd = typename TL0_TYPES::MCMD_T(d);}
  typename TL0_TYPES::DATA_T curr_mdata;
  void mdata(unsigned char d, unsigned byte_lane) {
    TL0_TYPES::data_from_uchar(d, byte_lane, curr_mdata);
  }
  void mdata() {MData = curr_mdata;}
  void mdatavalid(bool d) {MDataValid = typename TL0_TYPES::MDATAVALID_T(d);}
  void mrespaccept(bool d) {MRespAccept = typename TL0_TYPES::MRESPACCEPT_T(d);}
  unsigned char sdata(unsigned byte_lane) {
    return TL0_TYPES::data_to_uchar(SData->read(), byte_lane);
  }
  bool scmdaccept() {return tl1_tl0::port_to_bool(SCmdAccept);}
  bool sdataaccept() {return tl1_tl0::port_to_bool(SDataAccept);}
  unsigned sresp() {return tl1_tl0::port_to_unsigned(SResp);}

  // simple
  void maddrspace(unsigned d) {MAddrSpace = typename TL0_TYPES::MADDRSPACE_T(d);}
  void mbyteen(sc_dt::uint64 d) {MByteEn = typename TL0_TYPES::MBYTEEN_T(d);}
  void mdatabyteen(sc_dt::uint64 d) {MDataByteEn = typename TL0_TYPES::MDATABYTEEN_T(d);}
  void mdatainfo(tlm::tlm_generic_payload *txn, unsigned beat) {
    typename TL0_TYPES::MDATAINFO_T info;
    TL0_TYPES::conv_mdatainfo(info, *txn, beat);
    MDataInfo = info;
  }
  void mreqinfo(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::MREQINFO_T info;
    TL0_TYPES::conv_mreqinfo(info, *txn);
    MReqInfo = info;
  }
  void sdatainfo(tlm::tlm_generic_payload *txn, unsigned beat) {
    typename TL0_TYPES::SDATAINFO_T info = SDataInfo;
    TL0_TYPES::conv_sdatainfo(info, *txn, beat);
  }
  void srespinfo(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::SRESPINFO_T info = SRespInfo;
    TL0_TYPES::conv_srespinfo(info, *txn);
  }

  // burst
  void matomiclength(unsigned d) {MAtomicLength = typename TL0_TYPES::MATOMICLENGTH_T(d);}
  void mblockheight(unsigned d) {MBlockHeight = typename TL0_TYPES::MBLOCKHEIGHT_T(d);}
  void mblockstride(sc_dt::uint64 d) {MBlockStride = typename TL0_TYPES::MBLOCKSTRIDE_T(d);}
  void mburstlength(unsigned d) {MBurstLength = typename TL0_TYPES::MBURSTLENGTH_T(d);}
  void mburstprecise(bool d) {MBurstPrecise = typename TL0_TYPES::MBURSTPRECISE_T(d);}
  void mburstseq(unsigned d) {MBurstSeq = typename TL0_TYPES::MBURSTSEQ_T(d);}
  void mburstsinglereq(bool d) {MBurstSingleReq = typename TL0_TYPES::MBURSTSINGLEREQ_T(d);}
  void mdatalast(bool d) {MDataLast = typename TL0_TYPES::MDATALAST_T(d);}
  void mdatarowlast(bool d) {MDataRowLast = typename TL0_TYPES::MDATAROWLAST_T(d);}
  void mreqlast(bool d) {MReqLast = typename TL0_TYPES::MREQLAST_T(d);}
  void mreqrowlast(bool d) {MReqRowLast = typename TL0_TYPES::MREQROWLAST_T(d);}
  bool sresplast() {return tl1_tl0::port_to_bool(SRespLast);}
  bool sresprowlast() {return tl1_tl0::port_to_bool(SRespRowLast);}

  // tag
  void mdatatagid(unsigned d) {MDataTagID = typename TL0_TYPES::MDATATAGID_T(d);}
  void mtagid(unsigned d) {MTagID = typename TL0_TYPES::MTAGID_T(d);}
  void mtaginorder(bool d) {MTagInOrder = typename TL0_TYPES::MTAGINORDER_T(d);}
  unsigned stagid() {return tl1_tl0::port_to_unsigned(STagID);}
  bool staginorder() {return tl1_tl0::port_to_bool(STagInOrder);}

  // thread
  void mconnid(unsigned d) {MConnID = typename TL0_TYPES::MCONNID_T(d);}
  void mdatathreadid(unsigned d) {MDataThreadID = typename TL0_TYPES::MDATATHREADID_T(d);}
  void mthreadbusy(unsigned d) {MThreadBusy = typename TL0_TYPES::MTHREADBUSY_T(d);}
  void mthreadid(unsigned d) {MThreadID = typename TL0_TYPES::MTHREADID_T(d);}
  unsigned sdatathreadbusy() {return tl1_tl0::port_to_unsigned(SDataThreadBusy);}
  unsigned sthreadbusy() {return tl1_tl0::port_to_unsigned(SThreadBusy);}
  unsigned sthreadid() {return tl1_tl0::port_to_unsigned(SThreadID);}

  // sideband
  void merror(bool d) {MError = typename TL0_TYPES::MERROR_T(d);}
  void mflag(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::MFLAG_T flag;
    TL0_TYPES::conv_mflag(flag, *txn);
    MFlag = flag;
  }
  void mreset_n(bool d) {MReset_n = typename TL0_TYPES::MRESET_N_T(d);}
  bool serror() {return tl1_tl0::port_to_bool(SError);}
  const sc_core::sc_event &serror_event() {return SError->default_event();}
  void sflag(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::SFLAG_T flag = SFlag;
    TL0_TYPES::conv_sflag(flag, *txn);
  }
  const sc_core::sc_event &sflag_event() {return SFlag->default_event();}
  bool sinterrupt() {return tl1_tl0::port_to_bool(SInterrupt);}
  const sc_core::sc_event &sinterrupt_event() {return SInterrupt->default_event();}
  bool sreset_n() {return tl1_tl0::port_to_bool(SReset_n);}
  const sc_core::sc_event &sreset_event() {return SReset_n->default_event();}

  // transaction objects for thread-busy signalling
  tlm::tlm_generic_payload *get_tb_transaction() {
    return ocpTL1.get_tb_transaction();
  }
  tlm::tlm_generic_payload *get_flag_transaction() {
    return ocpTL1.get_tb_transaction();
  }
  tlm::tlm_generic_payload *get_error_transaction() {
    return ocpTL1.get_tb_transaction();
  }
  tlm::tlm_generic_payload *get_interrupt_transaction() {
    return ocpTL1.get_tb_transaction();
  }
  tlm::tlm_generic_payload *get_reset_transaction() {
    return ocpTL1.get_tb_transaction();
  }
  void ocp_socket_reset() {ocpTL1.reset();}

  bool eoe_called, timing_received;
  tl1_tl0::get_TL0_timing_base *get_TL0_timing;
};


// end ocpip namespace
}

