// 
//  (c) Copyright OCP-IP 2003, 2004
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//
//  Description : Layer Adapter, TL0-TL1 example, Asynchronous TL0 OCP Slave
//  $Id: ocp_tl0_slave_async.cpp,v 1.1.1.1 2004/09/24 09:26:13 sguntz Exp $
//
//    Features:
//    - Asynchronous TL0 slave implementing OCP2.0 Basic Signals 
//    - Propagation delays for output signals
//    
//  Parameter    :
//    Template arguments.
//     - TdataCl: Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type
//     - Ta: Address type
//                
//   Constructor arguments.
//     - sc_module_name: Name of the module instance
//     - sc_time RequestDelay:
//        delay between MCmd assertion and SCmdAccept assertion
//     - sc_time ResponseDelay:
//        delay between MCmd assertion and Response Group signals assertion
// ============================================================================

// Used for debug purposes
#define DEBUG_TL0_SLAVE_ASYNC

#include "ocp_tl0_slave_async.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL0_Slave_async::OCP_TL0_Slave_async
// 
// Constructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Slave_async<TdataCl>::OCP_TL0_Slave_async
       (
          sc_module_name name_
          , sc_time RequestDelay
          , sc_time ResponseDelay
        )
          : sc_module(name_)
          , m_RequestDelay(RequestDelay) 
          , m_ResponseDelay(ResponseDelay)
          , m_request_fifo(REQUEST_FIFO_DEPTH)
{
  // Test Constructor parameters
  if(ResponseDelay < RequestDelay) {
    cout << "Error: TL0 OCP Slave Constructor" << endl
      << "       ResponseDelay' parameter must be >= 'RequestDelay' parameter" << endl
      << "       ResponseDelay=" << ResponseDelay
      << "       RequestDelay=" << ResponseDelay
      << endl;
    assert(0);
  }
  
  // Output ports initializing
  SResp.initialize(sc_bv<2>(OCP_SRESP_NULL));
  SData.initialize(Td(0));
  SCmdAccept.initialize('0');
  
  // Request Processes
  SC_METHOD(MCmdThread);
  sensitive << MCmd;
  
  SC_METHOD(SCmdAcceptSetThread);
  dont_initialize();
  sensitive << e_SCmdAccept_Set ;
  
  SC_METHOD(SCmdAcceptResetThread);
  sensitive_pos << Clk;

  // Response Processes
  SC_METHOD(Request_delaying);
  dont_initialize();
  sensitive << m_request_fifo.data_written_event();

  SC_THREAD(ExecAndSendResponse);
  sensitive << e_Request_present_delayed;
   
} // end of constructor


// ----------------------------------------------------------------------------
// Process : OCP_TL0_Slave_async::~OCP_TL0_Slave_async
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_Slave_async<TdataCl>::~OCP_TL0_Slave_async()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_async::MCmdThread()
//
//  Catch 'MCmd' variations , push OCP request signals on the request fifo
//  and notify 'SCmdAccept' Thread (with a delay "m_RequestDelay")
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_async<TdataCl>::MCmdThread()
{
  if(MCmd.read() != sc_bv<3>( (OCPMCmdType)OCP_MCMD_IDLE))
  {

    // get Request signals 
    request_struct<Td,Ta> req;
    req.mcmd = MCmd.read().to_uint();
    req.mdata = MData.read();
    req.maddr = MAddr.read();

#ifdef DEBUG_TL0_SLAVE_ASYNC
    cout << "TL0 Slave got request "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds()
      << " addr " << req.maddr
      << " data " << req.mdata << endl;
#endif

    // push request 
    m_request_fifo.write(req);
    
    // notify event 'e_SCmdAccept_Set' 
    e_SCmdAccept_Set.notify(m_RequestDelay);
  } 
} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_async::SCmdAcceptSetThread()
//
//  Set 'SCmdAccept' signal 
//  
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_async<TdataCl>::SCmdAcceptSetThread()
{
  SCmdAccept.write(true);
#ifdef DEBUG_TL0_SLAVE_ASYNC
    cout << "TL0 Slave released request "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds()
      << endl;
#endif

} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_async::SCmdAcceptResetThread()
//
//  Reset 'SCmdAccept' signal 
//  
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_async<TdataCl>::SCmdAcceptResetThread()
{
  SCmdAccept.write(false);

} // end of method


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_async::Request_delaying()
//
//  Emits 'e_Request_present_delayed' event, sampled by SC_METHOD
//      ExecAndSendResponse() to issue responses 
//  Goal is to delay fifo 'data_written_event' event by an amount of time
//  equal to 'ResponseDelay'
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_async<TdataCl>::Request_delaying()
{

  e_Request_present_delayed.notify(m_ResponseDelay);

} // end of method

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_Slave_async::ExecAndSendResponse()
//  
//  Reads requests in fifo
//  Acts as a memory for READ and WRITE operations (after 'ResponseDelay' time)
//  Sends OCP responses for WRITE operations
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_Slave_async<TdataCl>::ExecAndSendResponse()
{
  while(true) {

    if(m_request_fifo.num_available()==0) {
      wait();       // wait next request notification by the request process
    } else {

      // Read request from fifo
      request_struct<Td,Ta> req = m_request_fifo.read();

      // Test Memory Address Overflow 
      if((int)req.maddr >= MEMORY_SIZE) {
        cout << "Error: TL0 OCP Slave 'ExecAndSendResponse' method" << endl
          << "       Request Address exceeds Memory size" << endl
          << "       Request Address=" << (int)req.maddr
          << "       MEMORY_SIZE=" << MEMORY_SIZE
          << endl;
        assert(0);
      }

      // Read Operation
      if(req.mcmd == OCP_MCMD_RD) {

#ifdef DEBUG_TL0_SLAVE_ASYNC
        cout << "TL0 Slave sent response "
          << " delta " << simcontext()->delta_count()
          << " time " << sc_time_stamp().to_seconds()
          << " addr " << (int)req.maddr
          << " data " << m_memory[(int)req.maddr] <<endl;
#endif


        // SResp & SData assertion 
        SResp.write(sc_bv<2>(OCP_SRESP_DVA)); 
        SData.write(m_memory[(int)req.maddr]);  

        // Wait end of current cycle and test 'MRespAccept'
        do {
          wait(Clk.posedge_event());
        } while ((MRespAccept.read() != true) );

        // SResp & SData de-assertion 
        SResp.write(sc_bv<2>(OCP_SRESP_NULL));
        SData.write(Td(0));

      } else if(req.mcmd == OCP_MCMD_WR) {
        // Write Operation
        m_memory[(int)req.maddr] = req.mdata;
#ifdef DEBUG_TL0_SLAVE_ASYNC
        cout << "TL0 Slave Write Operation "
          << " delta " << simcontext()->delta_count()
          << " time " << sc_time_stamp().to_seconds()
          << " addr " << (int)req.maddr
          << " data " << req.mdata <<endl;
#endif

      }

    } // end if 

  } // end of while

} // end of method


// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class OCP_TL0_Slave_async<OCP_TL1_DataCl<int, int> >;
