// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 13/01/2004
//  Description : Transaction Level - Layer-0 to Layer-1 Slave Adapter (OCP 2.0)
//  Version 1.0
//
//     Features :
//
//    - Layer adapter used to connect a TL0 OCP Slave to a TL1 system
//
//    - Uses the OCP-specific TL1 Channel and associated API
//
//    - Interface implements OCP Basic signals : Clk, MCmd, MAddr, MData,
//        SCmdAccept, SData, SResp & MRespAccept.
//
//    - Glitches and propagation delays for input signals are removed using two
//      'Sample delay' parameters.  Signals coming from the TL0 slave are not
//      supposed to be clocked and therefore can change at any time during the
//      clock cycle. It is also possible that some glitches appear on them,
//      particulary in the case of co-simulation with a HDL-simulated RTL
//      Slave. 
//
//      Once a TL1 interface call has been done, it can not be cancelled. The
//        slave adapter issues two kinds of call to the TL1 channel:
//          * startOCPResponse() 
//          * putSCmdAccept() 
//
//      The 'startOCPResponse()' and associated API calls must happen only
//        when TL0 associated signals 'SResp' and 'SData' are stable. Since the
//        propagation delays for these signals are slave dependant, a 'Response
//        Sample Delay' parameter is used to determine when these signals
//        should be sampled during the cycle.  Once sampled,'SResp' is tested;
//        if the response is valid (DVA value), a  'startOCPResponse()' call is done.
//
//      The 'putSCmdAccept()' call must happen only when 'SCmdAccept' is stable.
//        Since the propagation delay for this signal is slave dependant, a
//        'CmdAcceptSampleDelay' parameter is used to determine when
//        'SCmdAccept' should be sampled during the cycle.
//
//      The smallest value for the sample delays is sc_get_time_resolution() ;
//      it should be used in the case when TL0 master has pure 'zero-delay'
//      signals. Users should always use a timed sample delay (and not a
//      defined number of delta-cycles) since it is not possible to predict the
//      number of delta-cycles required for signal stability.
//
//      'Check_setup_time()' function is proposed for debugging purposes, to
//        ensure that TL0 slave output signals don't change after the sample
//        delays, which would cause the adapter to fail. A good way to proceed
//        is to perform the first simulation with checking function actived to
//        determine the appropriate sample delay value.  Once TL0 behaviour is
//        correct, checking function should be disabled to increase simulation
//        speed.
//                
// Parameters   : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//    First Constructor arguments (with explicit sampling times):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Slave.
//           ID must be unique among all Slaves attached to the same Bus.
//     - CmdAcceptSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for 'scmdaccept'
//                 signal
//     - ResponseSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for response
//                 signals
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//   Second Constructor arguments (sampling times use default values):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Combinational: Boolean specifying if Request/Response occurs in the
//                 same clock cycle or sequentially. Sampling times will be
//                 set as follows:
//                    - Combitional:
//                          * RequestSampleDelay = ClockCycle*99/100
//                          * ResponseSampleDelay = ClockCycle*60/100
//                    - Sequential:
//                          * RequestSampleDelay = ClockCycle*99/100
//                          * ResponseSampleDelay = ClockCycle*60/100
//     - ClockCycle: sc_time number specifying the clock cycle
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//     NOTE : The adapter uses events of the communication class to
//       synchronize.  TL1 channel connected to the slave port MUST have its
//       'syncevent' parameter set.
// ============================================================================

// Used for debug purposes
//#define DEBUG_SLAVE_ADAPTER

#include "ocp_tl0_tl1_slave_adapter.h"
#include "ocp_tl1_data_cl.h"

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Slave_Adapter::OCP_TL0_TL1_Slave_Adapter
// 
// Constructor with explicit sampling timings
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Slave_Adapter<TdataCl>::OCP_TL0_TL1_Slave_Adapter
(
 sc_module_name name_
 , int ID
 , sc_time CmdAcceptSampleDelay
 , sc_time ResponseSampleDelay
 , bool check_setup_time 
)
: sc_module         (name_)
    , SlaveP           ("SlavePort")
    , m_ID              (ID)
    , m_cmdaccept_sample_delay    (CmdAcceptSampleDelay)
    , m_response_sample_delay    (ResponseSampleDelay)
    , m_check_active    (check_setup_time)
, m_default_values  (false)
{

    // Check the sample delay values
    if(m_response_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'Response Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'Response Sample Delay' =" << m_response_sample_delay
            << endl;
        exit(-1);
    }

    if(m_cmdaccept_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'CmdAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'CmdAccept Sample Delay' =" << m_cmdaccept_sample_delay
            << endl;
        exit(-1);
    }


    // Init TL0 output signals:
    //    * 'MCmd' as IDLE
    //    * 'MRespAccept' as false 
    MCmd.initialize(OCP_MCMD_IDLE);
    MRespAccept.initialize(false);

    // Reset the response structure
    TL1_response.reset();

    // SC_METHODs
    SC_METHOD(send_Response);
    sensitive << e_response_sample_event;

    SC_METHOD(set_MRespAccept);
    sensitive << SlaveP.ResponseEndEvent();
    dont_initialize();

    SC_METHOD(reset_TL0_signals);
    sensitive_pos << Clk;

    SC_METHOD(send_Request);
    sensitive << SlaveP.RequestStartEvent();
    dont_initialize();

    SC_METHOD(release_Request);
    sensitive << e_cmdaccept_sample_event;

    SC_METHOD(sample_events_trigger);
    dont_initialize();
    sensitive_pos << Clk;

    if(m_check_active) {
        SC_METHOD(check_cmdaccept_setup_time);
        sensitive << SCmdAccept ;
        SC_METHOD(check_response_setup_time);
        sensitive << SData << SResp;
    }

}

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Slave_Adapter::OCP_TL0_TL1_Slave_Adapter
// 
// Constructor with default sampling timings.  
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Slave_Adapter<TdataCl>::OCP_TL0_TL1_Slave_Adapter
(
 sc_module_name name_
 , int ID
 , bool Combinational
 , sc_time ClockCycle
 , bool Check_setup_time

)
: sc_module         (name_)
    , SlaveP           ("SlavePort")
    , m_ID              (ID)
    , m_check_active    (Check_setup_time)
, m_default_values  (true)
{
    // Setting default sampling values
    if(Combinational) {
        m_cmdaccept_sample_delay = sc_time(ClockCycle*double(0.99));
        m_response_sample_delay = sc_time(ClockCycle*double(0.60));
    } else {
        m_cmdaccept_sample_delay = sc_time(ClockCycle*double(0.99));
        m_response_sample_delay = sc_time(ClockCycle*double(0.60));
    }

    if(m_response_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'Response Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'Response Sample Delay' =" << m_response_sample_delay
            << endl;
        exit(-1);
    }

    if(m_cmdaccept_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'CmdAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'CmdAccept Sample Delay' =" << m_cmdaccept_sample_delay
            << endl;
        exit(-1);
    }


    // Init 'MCmd' as IDLE and 'MRespAccept' as false 
    MCmd.initialize(OCP_MCMD_IDLE);
    MRespAccept.initialize(false);

    // Reset the response structure
    TL1_response.reset();

    // SC_METHODs
    SC_METHOD(send_Response);
    sensitive << e_response_sample_event;

    SC_METHOD(set_MRespAccept);
    sensitive << SlaveP.ResponseEndEvent();
    dont_initialize();

    SC_METHOD(reset_TL0_signals);
    sensitive_pos << Clk;

    SC_METHOD(send_Request);
    sensitive << SlaveP.RequestStartEvent();
    dont_initialize();

    SC_METHOD(release_Request);
    sensitive << e_cmdaccept_sample_event;

    SC_METHOD(sample_events_trigger);
    dont_initialize();
    sensitive_pos << Clk;

    if(m_check_active) {
        SC_METHOD(check_cmdaccept_setup_time);
        sensitive << SCmdAccept ;
        SC_METHOD(check_response_setup_time);
        sensitive << SData << SResp;
    }

}


// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Slave_Adapter::~OCP_TL0_TL1_Slave_Adapter
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Slave_Adapter<TdataCl>::~OCP_TL0_TL1_Slave_Adapter()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::send_Response()
//
//  Catch responses from the TL0 Slave, and send them to the TL1 channel 
//
//  Sensitivity: 
//      sensitive << e_response_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::send_Response()
{
    // Test if a valid response is present on TL0 signals
    if(!SlaveP->getMBusy() && (SResp.read() != sc_bv<2>(OCP_SRESP_NULL) ) ) {
        // Send a 'DVA' TL1 Response
        if(SResp.read() == sc_bv<2>(OCP_SRESP_DVA) ) {
            TL1_response.SResp = OCP_SRESP_DVA;
            TL1_response.SData = (Td)SData.read();
            bool ret = SlaveP->startOCPResponse(TL1_response);

            // For Debug purposes only
            assert(ret);
        } else {
            // Send an 'ERR' TL1 Response
            if(SResp.read() == sc_bv<2>(OCP_SRESP_ERR) ) {
                TL1_response.SResp = OCP_SRESP_ERR;
                TL1_response.SData = (Td)SData.read();
                bool ret = SlaveP->startOCPResponse(TL1_response);

                // For Debug purposes only
                assert(ret);
            } else {
                cout << "Error in 'OCP TL0 TL1 Slave Adapter' class 'send_Response' method " << endl
                    << "       TL0 SResp signal value '" << SResp.read() 
                    << "' is not supported" << endl;
                exit(-1);
            }
        }
    }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::set_MRespAccept()
//
//  Set 'MRespAccept' TL0 signal
//  
//  Sensitivity: 
//      sensitive << SlaveP.ResponseEndEvent();
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::set_MRespAccept()
{
    MRespAccept.write(true);
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::reset_TL0_signals()
//
//  Reset 'MRespAccept' and 'MCmd' TL0 signals 
//
//  Sensitivity: 
//      sensitive_pos << Clk;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::reset_TL0_signals()
{
    MRespAccept.write(false);
    if(!m_CommCl->RequestStart) {
        MCmd.write(OCP_MCMD_IDLE);
    }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::send_Request()
//
//  Catch TL1 request fields and send them to the TL0 slave 
//
//  Sensitivity: 
//  sensitive << SlaveP.RequestStartEvent();
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::send_Request()
{
    if(SlaveP->getOCPRequest(TL1_request)) {
        MCmd.write(TL1_request.MCmd);
        MData.write(TL1_request.MData);
        MAddr.write(TL1_request.MAddr);

#ifdef DEBUG_SLAVE_ADAPTER
        cout << "TL0 Slave Adapter sends a new request "
            << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_seconds()
            << " TL1_request.MCmd " << TL1_request.MCmd
            << " TL1_request.MAddr " << TL1_request.MAddr
            << " TL1_request.MData" << TL1_request.MData
            << endl;
#endif
    }

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::release_Request()
//
//  Release request present on TL1 channel
//
//  Sensitivity: 
//      sensitive << e_cmdaccept_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::release_Request()
{
    // NOTE: Access to the communication class: not very clean
    if(m_CommCl->RequestStart) { 
        if(SCmdAccept.read()) {
            SlaveP->putSCmdAccept(); 
        }
    }
} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::sample_events_trigger()
//
//  Trigger 'e_cmdaccept_sample_event' and 'e_response_sample_event'
//  used to sample input signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::sample_events_trigger()
{
#ifdef DEBUG_SLAVE_ADAPTER
    cout << endl
        << "TL0 Slave Adapter: CLOCK cycle"
        << " delta " << simcontext()->delta_count()
        << " time " << sc_time_stamp().to_seconds()
        << endl;
#endif

    e_cmdaccept_sample_event.notify(m_cmdaccept_sample_delay);
    e_response_sample_event.notify(m_response_sample_delay);

    if(m_check_active)
        m_last_rising_edge = sc_time_stamp();

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::check_response_setup_time()
//
//  Checks timing violations, i.e. if 'SResp' or 'SData' signals
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::check_response_setup_time()
{
    if( (sc_time_stamp()-m_last_rising_edge) >= m_response_sample_delay ) {
        cout << "TL0 Slave Adapter '" << name() << "': Setup Time Violation "
            << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_seconds() << endl
            << " last rising edge " << m_last_rising_edge.to_seconds() << endl
            << " one of the signal 'SResp' or 'SData' had a late change" << endl
            << " You have to adjust 'ResponseSampleDelay' parameter" 
            << endl;

        if(m_default_values) {
            cout << " Default values for sampling times do not work with that design." << endl
                << " You have to explicitly specify sampling times using the appropriate constructor" 
                << endl;
        }

        sc_stop();
    }

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::check_cmdaccept_setup_time()
//
//  Checks timing violations, i.e. if 'SCmdAccept' signal
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::check_cmdaccept_setup_time()
{
    if( (sc_time_stamp()-m_last_rising_edge) >= m_cmdaccept_sample_delay ) {
        cout << "TL0 Slave Adapter '" << name() << "': Setup Time Violation "
            << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_seconds() << endl
            << " last rising edge " << m_last_rising_edge.to_seconds() << endl
            << " signal 'SCmdAccept' had a late change" << endl
            << " You have to adjust 'CmdAcceptSampleDelay' parameter" 
            << endl;

        if(m_default_values) {
            cout << " Default values for sampling times do not work for that design." << endl
                << " You have to explicitly specify sampling times using the appropriate constructor" 
                << endl;
        }

        sc_stop();
    }

} // end of SC_METHOD 



// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Async::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL0_TL1_Slave_Adapter<TdataCl>::MputDirect(
        int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{

    // not implemented
    return(false);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter<TdataCl>::end_of_elaboration()
{
    sc_module::end_of_elaboration();

    // Get communication structure
    m_CommCl = SlaveP->GetCommCl();

    // Put parameter into Channel
    // No parameters needed
}


// ----------------------------------------------------------------------------
//  Instantiation of the Master
// ----------------------------------------------------------------------------
template class OCP_TL0_TL1_Slave_Adapter<OCP_TL1_DataCl<int, int> >;
