// 
//  Copyright 2004 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 01/05/2004
//
//  Description : 
//  testbench for tl2-tl1 slave adapter example
//
//  The master sends the following sequence of requests:
//
//	- a WRITE burst composed of one chunk: length=20, start address=0
//	- a READ burst composed of one chunk:  length=10, start address=0
//	then it waits for 2 nanoseconds
//	- a READ burst composed of two chunks: total length=8, first chunk length=4, second chunk length=4,
//	(start address=0 for each of the chunk)
//	the master waits for 2 nanoseconds after the first chunk, and 3 nanoseconds after the second chunk
//	- a WRITE burst composed of one chunk: length=15, start address=0
//
// On the response side, the master releases the response channel after a 2 nanosecond waiting period
//
// The TL1-TL2 slave adapter separates the first burst into 20 consecutive TL1 write requests (written data are 0 to 19).
// The following TL2 READ burst is decomposed into 10 consecutive TL1 READ requests (start address=0, end address=36).
// The 10 TL1 responses are sent to the master as two chunks (size=5, which is the maximum chunk length).
// For the first chunk, the read data are 0,1,2,3,4. For the second chunk, the read data are 5,6,7,8,9.
//
// For the second READ burst (total length=8), the adapter sends 8 TL1 READ requests (@address=0,4,8,12 for the first chunk
// and @address=0,4,8,12 for the second chunk)
//
// The adapter gets 8 responses which are sent to the master as two chunks. For the first chunk (size=5), read data are  
// 0,1,2,3,0. The second chunk has a size of 3 and read data are 1,2,3.
//
// Then the adapter ends with sending 15 write requests to the slave (written data are 0 to 14)  
// 
// ============================================================================

#include "systemc.h"
#include "simple_ocp_master_tl2.h"
#include "ocp_tl1_slave_sync.h"
#include "ocp_tl1_tl2_slave_adapter.h"
#include "ocp_tl1_channel.h"
#include "ocp_tl2_channel.h"

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}



int sc_main(int argc, char* argv[])
{
  
  sc_clock clk("clock", 1, SC_NS, 0.5, 0, SC_NS, true);

   //TL1 and TL2 channels instantiation 
  OCP_TL1_Channel<OCP_TL1_DataCl<int , int> > *channel0=new OCP_TL1_Channel<OCP_TL1_DataCl<int , int> >("channel0");
  OCP_TL2_Channel< int , int> *channel1=new OCP_TL2_Channel< int , int >("channel1");

   // Set the OCP parameters for this channel
  MapStringType  ocpParamMap;
  readMapFromFile("ocpParams_slave_adapter_tl1_tl2", ocpParamMap);
  channel1->setConfiguration(ocpParamMap);
  channel0->setConfiguration(ocpParamMap);

  //TL2 master and TL1 slave instantiation
  simple_ocp_master_tl2 *master1=new simple_ocp_master_tl2("master1"); 
  OCP_TL1_Slave_Sync<OCP_TL1_DataCl<int , int> > *slave1=new OCP_TL1_Slave_Sync<OCP_TL1_DataCl<int , int> >("slave1", 2, 0, 0xff);

   //adapter instantiation
  //chunk length=5, maximum burst length=25, adapter depth= 6
  OCP_TL1_TL2_Slave_Adapter<OCP_TL1_DataCl<int , int>, OCP_TL2_DataCl<int , int> > *slave_adapter=new OCP_TL1_TL2_Slave_Adapter<OCP_TL1_DataCl<int , int>, OCP_TL2_DataCl<int , int> >("slave_adapter", 5, 25, 6);

  //binding
  master1->MasterP(*channel1);

  slave_adapter->MasterP(*channel0);
  slave_adapter->SlaveP(*channel1);
  slave_adapter->clk(clk);

  slave1->SlaveP(*channel0);
  slave1->clk(clk);

  sc_start(200, SC_NS);


  return 0;
}
