// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//           $Id: slave.cc,v 1.1 2005/01/07 03:42:33 Anssi Exp $
//
//  Description : OCP API - TL1 profile example
//                Simple 2-port slave with no real arbitration
//                ReadEx is not implemented.
// ============================================================================

#include "slave.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
Slave::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), wP("wPort"), rP("rPort")
  , req_w(NULL)
  , req_r(NULL)
  , req_data(NULL)
  , ongoing_resp_w(false)
  , ongoing_resp_r(false)
//  , is_srmd_r(false) 
{

  // initialize common members
  for (int i=0;i<1024;i++)
    memory[i] = 0;

  address_w=0;

  SC_METHOD(proc_w);
  sensitive<<clk.pos();
  dont_initialize(); 

  SC_METHOD(proc_r);
  sensitive<<clk.pos();
  dont_initialize(); 

  wP.register_nb_transport_fw(this, &Slave::nb_transport_w);
  rP.register_nb_transport_fw(this, &Slave::nb_transport_r);
  wP.activate_synchronization_protection();
  rP.activate_synchronization_protection();

  current_txn_r=current_txn_w=NULL;
  ongoing_srmd_r=NULL;
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
Slave::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::proc_w()
//
//  Synchronous slave write port process
//
// ----------------------------------------------------------------------------

void Slave::proc_w(){

  if (req_w){
#ifdef DEBUG_G1
  std::cout << "Slave got write request "
	 << " time " << sc_core::sc_time_stamp().to_seconds()
	 << " MCmd " << req_w->get_command()
	 << " MAddr " << req_w->get_address() << std::endl;
#endif

    if (pending_reqs_w.size()<8) { //we can still accept requests
      // Burst addressing
      if (current_txn_w!=req_w) {
        //since we react to a posted write just as we react to a nonposted one, we 
        // can ignore the posted extension      
        current_txn_w=req_w;
        chunk_cnt_w=0;
        address_w=req_w->get_address()/4; // First of burst
        ocpip::burst_sequence* b_seq;
        ocpip::burst_length* b_len;
        burst_length_w=1;
        if (wP.get_extension<ocpip::burst_length>(b_len, *req_w)) burst_length_w=b_len->value;
        current_seq_w=ocpip::INCR;
        if (wP.get_extension<ocpip::burst_sequence>(b_seq, *req_w)) current_seq_w=b_seq->value.sequence;
        if (current_seq_w==ocpip::WRAP){
          address_w=b_seq->value.xor_wrap_address/4;
          pending_reqs_w.push_back(pending_req(req_w, address_w, address_w-(~(burst_length_w-1) & address_w)));
        }
        else
          pending_reqs_w.push_back(pending_req(req_w, address_w, 0));
      }
      else {
        chunk_cnt_w++;
        if (current_seq_w==ocpip::INCR){
          address_w = address_w+1; //incrementing address
          pending_reqs_w.push_back(pending_req(req_w, address_w, chunk_cnt_w));
        }
        else 
        if (current_seq_w==ocpip::STRM){
          address_w = address_w; //streaming address
          pending_reqs_w.push_back(pending_req(req_w, address_w, chunk_cnt_w));
        }
        else 
        if (current_seq_w==ocpip::WRAP) {
          if ((address_w+1) < ((address_w & (~(burst_length_w-1)))+ burst_length_w)) // wrapping address
            address_w = address_w+1; 
          else
            address_w = address_w & (~(burst_length_w-1));
          pending_reqs_w.push_back(pending_req(req_w, address_w, address_w-(~(burst_length_w-1) & address_w)));
        }
        else assert(0);
        
      }

      time=sc_core::SC_ZERO_TIME;
      phase=tlm::END_REQ;
      tlm::tlm_sync_enum retVal=wP->nb_transport_bw(*req_w,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_REQ"<<std::endl;
            exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
  #ifdef DEBUG_G1
      std::cout << "Slave accepted write request "
           << " time " << sc_core::sc_time_stamp().to_seconds() << std::endl;
  #endif
      req_w=NULL;
    }
  }

  // Data handshake
  if (pending_reqs_w.size() && (outstanding_rsps_w.size()<8)) {
    if (req_data) {
      assert(req_data==pending_reqs_w.front().txn);
      memory[pending_reqs_w.front().address] = (*(((Td*)(req_data->get_data_ptr()))+pending_reqs_w.front().chunk));

    #ifdef DEBUG_G1
        std::cout << "Slave got data "
             << " time " << sc_core::sc_time_stamp().to_seconds()
             << " MData " << (*(((Td*)(req_data->get_data_ptr()))+(pending_reqs_w.front().chunk))) << std::endl;
    #endif
      outstanding_rsps_w.push_back(outstanding_rsp(req_data, 0, pending_reqs_w.front().chunk));
      pending_reqs_w.pop_front();
      time=sc_core::SC_ZERO_TIME;
      phase=ocpip::END_DATA;
      tlm::tlm_sync_enum retVal=wP->nb_transport_bw(*req_data,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_DATA"<<std::endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
      req_data=NULL;
    }
  }

  if (outstanding_rsps_w.size()) {
    if (!ongoing_resp_w) {
      // Set OCP response
      outstanding_rsps_w.front().txn->set_response_status(tlm::TLM_OK_RESPONSE);
      // Send response
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::BEGIN_RESP;
      tlm::tlm_sync_enum retVal=wP->nb_transport_bw(*outstanding_rsps_w.front().txn,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: ongoing_resp_w=true; break;
        case tlm::TLM_UPDATED:
          if (phase!=tlm::END_RESP) {
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to BEGIN_RESP"<<std::endl;
            exit(1);
          }
          break;
        case tlm::TLM_COMPLETED: break; //we don't check that it was really the last resp. we just trust the master...
      }


#ifdef DEBUG_G1
      std::cout << "Slave sent response "
	   << " time " << sc_core::sc_time_stamp().to_seconds()
	   << " SResp " << outstanding_rsps_w.front().txn->get_response_status() << std::endl;
#endif      
      outstanding_rsps_w.pop_front();
    }
  }
} // end of method

tlm::tlm_sync_enum Slave::nb_transport_w(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
  switch(ph){
    case tlm::BEGIN_REQ: req_w=&txn;break;
    case tlm::END_RESP: ongoing_resp_w=false; break;
    default:
      if (ph==ocpip::BEGIN_DATA){
        req_data=&txn;
      }
      else{
        std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
        exit(1);        
      }
  }
  return tlm::TLM_ACCEPTED;
}


// ----------------------------------------------------------------------------
//  Method : Slave::proc_r()
//
//  Synchronous slave write port process
//
// ----------------------------------------------------------------------------

void Slave::proc_r(){


  // Request handshake
  if (req_r){
#ifdef DEBUG_G1
      std::cout<< "Slave got read request "
               << " time " << sc_core::sc_time_stamp().to_seconds()
               << " MCmd " << req_r->get_command()
               << " MBurstSingleReq " << rP.get_extension<ocpip::srmd>(*req_r)//is_srmd_r
               << " MAddr " << req_r->get_address() << std::endl;
#endif
    if (outstanding_rsps_r.size()<8){
      // Burst addressing
      if (current_txn_r!=req_r ) {
        current_txn_r=req_r;
        chunk_cnt_r=0;
        address_r=req_r->get_address()/4; // First of burst
        ocpip::burst_sequence* b_seq;
        ocpip::burst_length* b_len;
        burst_length_r=1;
        if (rP.get_extension<ocpip::burst_length>(b_len, *req_r)) burst_length_r=b_len->value;
        current_seq_w=ocpip::INCR;
        if (rP.get_extension<ocpip::burst_sequence>(b_seq, *req_r)) current_seq_r=b_seq->value.sequence;
        if (rP.get_extension<ocpip::srmd>(*req_r))
          ongoing_srmd_r=current_txn_r;
        if (current_seq_r==ocpip::WRAP){
          address_r=b_seq->value.xor_wrap_address/4;
          outstanding_rsps_r.push_back(outstanding_rsp(req_r, address_r, address_r-(~(burst_length_r-1) & address_r)));
        }
        else          
          outstanding_rsps_r.push_back(outstanding_rsp(req_r, address_r, 0));
      }
      else {
        chunk_cnt_r++;
        if (current_seq_r==ocpip::INCR){
          address_r = address_r+1; //incrementing address
          outstanding_rsps_r.push_back(outstanding_rsp(req_r, address_r, chunk_cnt_r));
        }
        else 
        if (current_seq_r==ocpip::STRM){
          address_r = address_r; //streaming address
          outstanding_rsps_r.push_back(outstanding_rsp(req_r, address_r, chunk_cnt_r));
        }
        else
        if (current_seq_r==ocpip::WRAP){
          if ((address_r+1) < ((address_r & (~(burst_length_r-1)))+ burst_length_r)) // wrapping address
            address_r = address_r+1; 
          else
            address_r = address_r & (~(burst_length_r-1));
          outstanding_rsps_r.push_back(outstanding_rsp(req_r, address_r, address_r-(~(burst_length_r-1) & address_r)));
        }
        else
          assert(0);
      }

      if (!ongoing_srmd_r && outstanding_rsps_r.size()<8){
        time=sc_core::SC_ZERO_TIME;
        //if (is_srmd_r) phase=ocpip::END_SRMD_REQ; else 
        phase=tlm::END_REQ;
        tlm::tlm_sync_enum retVal=rP->nb_transport_bw(*req_r,phase, time);
        switch(retVal){
          case tlm::TLM_ACCEPTED: break;
          case tlm::TLM_UPDATED:
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_(SRMD_)REQ"<<std::endl;
            exit(1);
            break;
          case tlm::TLM_COMPLETED:;
        }
  #ifdef DEBUG_G1
        std::cout << "Slave accepted"<<(rP.get_extension<ocpip::srmd>(*req_r)?" SRMD1":"")<<" read request "
             << " time " << sc_core::sc_time_stamp().to_seconds() << std::endl;
  #endif
      }
      req_r=NULL;
    }
  }
  else{
    if (ongoing_srmd_r && outstanding_rsps_r.size()<8){
      chunk_cnt_r++;
      if (current_seq_r==ocpip::INCR){
        address_r = address_r+1; //incrementing address
        outstanding_rsps_r.push_back(outstanding_rsp(ongoing_srmd_r, address_r, chunk_cnt_r));
      }
      else 
      if (current_seq_r==ocpip::STRM){
        address_r = address_r; //streaming address
        outstanding_rsps_r.push_back(outstanding_rsp(ongoing_srmd_r, address_r, chunk_cnt_r));
      }
      else
      if (current_seq_r==ocpip::WRAP){
        if ((address_r+1) < ((address_r & (~(burst_length_r-1)))+ burst_length_r)) // wrapping address
          address_r = address_r+1; 
        else
          address_r = address_r & (~(burst_length_r-1));
        outstanding_rsps_r.push_back(outstanding_rsp(ongoing_srmd_r, address_r, address_r-(~(burst_length_r-1) & address_r)));
      }
      else
        assert(0);
      
    
      if (chunk_cnt_r==burst_length_r-1){
        time=sc_core::SC_ZERO_TIME;
        //phase=ocpip::END_SRMD_REQ;
        phase=tlm::END_REQ;
        tlm::tlm_sync_enum retVal=rP->nb_transport_bw(*ongoing_srmd_r,phase, time);
        switch(retVal){
          case tlm::TLM_ACCEPTED: break;
          case tlm::TLM_UPDATED:
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_REQ"<<std::endl;
            exit(1);
            break;
          case tlm::TLM_COMPLETED:;
        }
  #ifdef DEBUG_G1
        std::cout << "Slave accepted SRMD read request "
             << " time " << sc_core::sc_time_stamp().to_seconds() << std::endl;
  #endif
        ongoing_srmd_r=NULL;
      }
    }
  }

  //NOTE: there is no mutual exclusion between write and read here!!! Not recommended!
  if (outstanding_rsps_r.size() && address_w!=outstanding_rsps_r.front().address) {
    if (!ongoing_resp_r) {
      // Set OCP response
      outstanding_rsps_r.front().txn->set_response_status(tlm::TLM_OK_RESPONSE);
      *((Td*)(outstanding_rsps_r.front().txn->get_data_ptr())+(outstanding_rsps_r.front().chunk))=memory[outstanding_rsps_r.front().address];
      // Send response
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::BEGIN_RESP;
      tlm::tlm_sync_enum retVal=rP->nb_transport_bw(*outstanding_rsps_r.front().txn,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: ongoing_resp_r=true; break;
        case tlm::TLM_UPDATED:
          if (phase!=tlm::END_RESP) {
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to BEGIN_RESP"<<std::endl;
            exit(1);
          }
          break;
        case tlm::TLM_COMPLETED: break; //we don't check that it was really the last resp. we just trust the master...
      }
#ifdef DEBUG_G1
      std::cout << "Slave sent read response "
	   << " time " << sc_core::sc_time_stamp().to_seconds()
	   << " SData " << *((Td*)(outstanding_rsps_r.front().txn->get_data_ptr())+(outstanding_rsps_r.front().chunk))       
	   << " SResp " << outstanding_rsps_r.front().txn->get_response_status() << std::endl;
#endif      
      outstanding_rsps_r.pop_front();
    }
  }
} // end of method

tlm::tlm_sync_enum Slave::nb_transport_r(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
  switch(ph){
    case tlm::BEGIN_REQ: req_r=&txn; break; //is_srmd_r=0; 
    case tlm::END_RESP: ongoing_resp_r=false; break;
    default:
      /*
      if (ph==ocpip::BEGIN_SRMD_REQ) {
        req_r=&txn; is_srmd_r=1;
      }
      else{
      */
        std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
        exit(1);        
      //}
  }
  return tlm::TLM_ACCEPTED;
}
