/*
  James Aldis
  TI France
  OCP TL1 Channel Timing Distribution
  29 June 2005

  C++ Header File for:
    multi-threaded thread-busy-exact test request-merger/response splitter
*/


// multiple inclusion protection
#ifndef TL1_TIMING_MERGER_H
#define TL1_TIMING_MERGER_H


#include "tl1_timing_common.h"


class tl1_timing_merger : 
  public sc_module,
  public OCP_TL1_Slave_TimingIF,
  public OCP_TL1_Master_TimingIF
{
  SC_HAS_PROCESS(tl1_timing_merger);
  public:
    tl1_timing_merger(sc_module_name name, int threads_Ap, int threads_Bp) :
      sc_module(name), ocpsA("ocpsA"), ocpsB("ocpsB"), ocpm("ocpm"),
      threads_A(threads_Ap), threads_B(threads_Bp) {

      SC_METHOD(clock_rising);
      sensitive(clk.pos());
      dont_initialize();

      SC_METHOD(request_arb);
      sensitive(arbiter_event);
      dont_initialize();

      SC_METHOD(mthreadbusy_propagate);
      sensitive(mthreadbusy_event);
      dont_initialize();

      SC_METHOD(response_split);
      sensitive(ocpm.ResponseStartEvent());
      dont_initialize();

      req_reg = new OCP_REQ[threads_A + threads_B];
      reqA_reg = req_reg;
      reqB_reg = &(req_reg[threads_A]);

      time_quantum = sc_get_time_resolution();
      sthreadbusy_sample_time = time_quantum;  // initial guess
      request_sample_time = time_quantum;  // initial guess
      mthreadbusy_sample_time = time_quantum;  // initial guess
    };

    ~tl1_timing_merger() {
      cout << "Deleting merger:   " << name() << endl;

      for(int i=0; i<threads_A; i++) {
        cout << (reqA_reg[i].MCmd != OCP_MCMD_IDLE ? "A " : "- ");
      }
      cout << endl;
      for(int i=0; i<threads_B; i++) {
        cout << (reqB_reg[i].MCmd != OCP_MCMD_IDLE ? "B " : "- ");
      }
      cout << endl;
      cout << "  SThreadBusy sample time: " << sthreadbusy_sample_time << endl;
      cout << "  MThreadBusy sample time: " << mthreadbusy_sample_time << endl;
      cout << "  Request sample time:     " << request_sample_time << endl;
      cout << "  Response input time:     " << response_in_time << endl;
    }

    // ports
    sc_in<bool> clk;
    OCP_SLAVE_PORT ocpsA, ocpsB;
    OCP_MASTER_PORT ocpm;

    // processes
    void clock_rising() {

      // generate SThreadBusy for both slave ports based on
      // loading of request registers
      int stb_out = 0;
      int mask = 1;
      for(int i=0; i<(threads_A+threads_B); i++) {
        if(req_reg[i].MCmd != OCP_MCMD_IDLE) {
          stb_out |= mask;
        }
        mask <<= 1;
      }
      ocpsA->putSThreadBusy(stb_out & ((1 << threads_A)-1));
      ocpsB->putSThreadBusy(stb_out >> threads_A);

      arbiter_event.notify(sc_max(request_sample_time, sthreadbusy_sample_time));

      // response side:  get thread busy from both masters and give to
      // slave.  responses are passed to the appropriate master (based on thread
      // ID), as they arrive.
      // ** this is not needed once we have MThreadBusy value change events **
      // but we will still need to tell other modules our timing, which depends
      // on input timing.
      mthreadbusy_event.notify(mthreadbusy_sample_time);
    }

    // the response path is really simple - thread-busy-signals are
    // concatencated and passed to slave.  slave returns a response which
    // is forwarded to the right slave port
    void mthreadbusy_propagate() {
      int mtb_A = ocpsA->getMThreadBusy();
      int mtb_B = ocpsB->getMThreadBusy();
      ocpm->putMThreadBusy(mtb_A | (mtb_B << threads_A));
    }

    void response_split() {
      OCP_RESP tmp;
      ocpm->getOCPResponse(tmp);
      if(int(tmp.SThreadID) < threads_A) {
        ocpsA->startOCPResponse(tmp);
      } else {
        tmp.SThreadID -= threads_A;
        ocpsB->startOCPResponse(tmp);
      }
    }

    void request_arb() {
      // we need registers for requests from all threads to allow both
      // slave ports to set sthreadbusy[x]=0 at the same time.  these are
      // bypassable.
      // we know that the register is free if there is a new request
      // on a thread, so we can use it as temporary storage without
      // a systematic 1 cycle latency
      int stb = ocpm->getSThreadBusy();

      OCP_REQ tmp;
      if(ocpsA->getOCPRequest(tmp)) {
        req_reg[tmp.MThreadID] = tmp;
      }
      if(ocpsB->getOCPRequest(tmp)) {
        tmp.MThreadID += threads_A;
        req_reg[tmp.MThreadID] = tmp;
      }
      // now arbitrate: priority is smallest thread ID, A then B
      int mask = 1;
      for(int i=0; i<(threads_A+threads_B); i++) {
        if((req_reg[i].MCmd != OCP_MCMD_IDLE) && !(mask & stb)) {
          // grant this one
          ocpm->startOCPRequest(req_reg[i]);
          req_reg[i].MCmd = OCP_MCMD_IDLE;
          break;
        }
        mask <<= 1;
      }
    }

    void end_of_elaboration() {
      cout << "<<<< E-O-E >>>> " << name() << endl;

      // merger is not default-timing and therefore must inform the
      // channel of its actual timing:
      //   request must wait for SThreadBusy and request inputs
      //   MThreadBusy must wait for MThreadBusy in
      //   response must wait for response in

      // calculate start times from sample times (sample times are local
      // variables of this module) and inform all OCP ports
      OCP_TL1_Master_TimingCl my_mtiming;
      my_mtiming.RequestGrpStartTime =
                sc_max(sthreadbusy_sample_time, request_sample_time);
      my_mtiming.MThreadBusyStartTime = mthreadbusy_sample_time;
      ocpm->setOCPTL1MasterTiming(my_mtiming);

      OCP_TL1_Slave_TimingCl my_stiming;
      my_stiming.ResponseGrpStartTime = response_in_time + time_quantum;
      ocpsA->setOCPTL1SlaveTiming(my_stiming);
      ocpsB->setOCPTL1SlaveTiming(my_stiming);

      // merger is timing-sensitive on slave and master ports.
      // fortunately it has the same timing on both slave ports,
      // so we do not need a sub-class definition to distinguish them.
      ocpm->registerTimingSensitiveOCPTL1Master(this);
      ocpsA->registerTimingSensitiveOCPTL1Slave(this);
      ocpsB->registerTimingSensitiveOCPTL1Slave(this);
    }

    // when informed of master port timing, merger must re-inform the OCP
    // channels if anything changed
    void setOCPTL1SlaveTiming(OCP_TL1_Slave_TimingCl slave_timing) {
      cout << "  << S-S-T >>   " << name() << endl;

      // if sthreadbusy start changed, may need to change request timing.

      // calculate the sample time for sthreadbusy based on the new timing
      // information from the channel
      sc_time stb_sample = slave_timing.SThreadBusyStartTime + time_quantum;
      // if larger than before, update the local sample time configuration
      // and, only if the request start time increased, inform the OCP port
      // (the request start time also depends on the request sample time)
      if(stb_sample > sthreadbusy_sample_time) {
        sc_time old_req_start =
                  sc_max(sthreadbusy_sample_time, request_sample_time);
        sthreadbusy_sample_time = stb_sample;
        sc_time new_req_start =
                  sc_max(sthreadbusy_sample_time, request_sample_time);
        if(new_req_start > old_req_start) {
          OCP_TL1_Master_TimingCl my_mtiming;
          my_mtiming.RequestGrpStartTime = new_req_start;
          my_mtiming.MThreadBusyStartTime = mthreadbusy_sample_time;
          ocpm->setOCPTL1MasterTiming(my_mtiming);
        }
      }

      // if response start increased, the OCP slave ports need to be informed
      // that the response timing increased
      if(slave_timing.ResponseGrpStartTime > response_in_time) {
        response_in_time = slave_timing.ResponseGrpStartTime;
        OCP_TL1_Slave_TimingCl my_stiming;
        my_stiming.ResponseGrpStartTime = response_in_time + time_quantum;
        ocpsA->setOCPTL1SlaveTiming(my_stiming);
        ocpsB->setOCPTL1SlaveTiming(my_stiming);
      }
    }

    // when informed of master timing, merger must re-inform the OCP
    // channels if anything changed
    void setOCPTL1MasterTiming(OCP_TL1_Master_TimingCl master_timing) {
      cout << "  << S-M-T >>   " << name() << endl;

      // start by calculating current (master) timing
      OCP_TL1_Master_TimingCl my_mtiming;
      my_mtiming.RequestGrpStartTime =
                sc_max(sthreadbusy_sample_time, request_sample_time);
      my_mtiming.MThreadBusyStartTime = mthreadbusy_sample_time;
      bool changed = false;

      // if request sample time increased this must be recorded
      sc_time new_req_sample = master_timing.RequestGrpStartTime + time_quantum;
      if(new_req_sample > request_sample_time) {
        request_sample_time = new_req_sample;

        // if request output time increased (also depends on sthreadbusy
        // sample time) the channel must be informed
        sc_time new_req_start = sc_max(sthreadbusy_sample_time,
                  master_timing.RequestGrpStartTime + time_quantum);
        if(new_req_start > my_mtiming.RequestGrpStartTime) {
          changed = true;
          my_mtiming.RequestGrpStartTime = new_req_start;
        }
      }

      // if mthreadbusy sample time increased, mthreadbusy-output time
      // will increase and the channel must be informed
      sc_time new_mtb_sample = master_timing.MThreadBusyStartTime +
                time_quantum;
      if(new_mtb_sample > mthreadbusy_sample_time) {
        changed = true;
        mthreadbusy_sample_time = new_mtb_sample;
        my_mtiming.MThreadBusyStartTime = new_mtb_sample;
      }

      if(changed) {
        ocpm->setOCPTL1MasterTiming(my_mtiming);
      }
    }

  private:
    sc_time time_quantum;
    sc_time sthreadbusy_sample_time;
    sc_time request_sample_time;
    sc_time mthreadbusy_sample_time;
    sc_time response_in_time;

    sc_event arbiter_event, mthreadbusy_event;
    OCP_REQ *req_reg, *reqA_reg, *reqB_reg;
    int threads_A, threads_B;
};


// end of multiple inclusion protection
#endif

