interface i2c_if(inout wire sda, input wire scl);
timeunit 1ns;
timeprecision 1ps;

import state_vars_pkg::*;
import uvm_pkg::*;
`include "uvm_macros.svh"

bit sda_drive; // intermediary registers for sda and scl
integer ii = 0;
integer jj = 0;
integer sda_delay = 10;

bit [7:0] reg_out, reg_monitor; // byte-sized registers for reading and writing

i2c_state_t i2c_state; // state variable (probe for debugging)
ack_debug_t ack_debug; // for storing ack at an instant
ack_debug_t ack_return = nack; // only becomes ack at posedge of relevant clock cycle

int fail_count = 0;

assign (pull1, strong0) sda = (sda_drive === 1'b0) ? 1'b0 : 1'bz;
// scl driven from hardware top

	task send_to_dut( bit [7:0] regAddr,
                  	bit [7:0] data,
                  	bit [6:0] id,
                  	bit       rw_); // rw_ used as data object first      
  	if (!rw_)  begin
  	  sendDataToTarget(.idToSend(id),
											 .regAddrToSend(regAddr),
											 .dataToSend(data));
  	end
  	else begin 
			readDataFromTarget(.idToSend(id),
												 .regAddrToRead(regAddr));
  	end
	endtask 

	task collect_pkt( output bit [7:0] reg_addr, 
                    output bit [7:0] data,
                    output bit [6:0] tg_addr,
                    output bit       rw_); 
		
    `uvm_info("i2c_if", "Initiator writing ID and rw_ ...", UVM_HIGH)
    read_byte();
    {tg_addr, rw_} = reg_monitor;
    if (ack_return == nack) begin 
      `uvm_info("i2c_if", "Invalid target ID or rw_ detected", UVM_MEDIUM)
      disable collect_pkt;
    end else begin 
      `uvm_info("i2c_if", "ACK detected after reading target address", UVM_HIGH)
    end

    `uvm_info("i2c_if", "Initiator writing register address to target ...", UVM_HIGH)
    read_byte();
    reg_addr = reg_monitor;
    if (ack_return == nack) begin 
      `uvm_info("i2c_if", "Invalid register address detected", UVM_MEDIUM)
      disable collect_pkt;
    end else begin
      `uvm_info("i2c_if", "ACK detected from target after reading register address", UVM_HIGH)
    end

    @(negedge scl); 
    ack_return = nack;
    
    @(posedge scl);
    reg_monitor = 8'b0;
    reg_monitor[7] = sda; // sample the first bit just in case it is a 

    fork
      begin : read_byte_b
        @(negedge sda iff scl); // repeated start condition
        disable write_byte_b;
        `uvm_info("i2c_if", "Repeated start condition detected from monitor task call.", UVM_HIGH)
        `uvm_info("i2c_if", "Initiator writing ID and rw_ ...", UVM_HIGH)
        read_byte();
        {tg_addr, rw_} = reg_monitor;
        if (ack_return == nack) begin 
          `uvm_info("i2c_if", "Invalid target ID or rw_ detected after repeated start.", UVM_MEDIUM)
          disable collect_pkt;
        end else begin
          `uvm_info("i2c_if", "ACK detected from target after repeated start.", UVM_HIGH)
        end

        `uvm_info("i2c_if", "Initiator reading data from target ...", UVM_HIGH)
        read_byte();  
        data = reg_monitor;
        if (ack_return == nack) begin 
          `uvm_info("i2c_if", "NACK detected after initiator correctly read data from target.", UVM_HIGH) // high because it is confusing (NACK is what is supposed to happen)
          `uvm_info("i2c_if", "Read transaction successful!", UVM_MEDIUM)
        end else begin
          `uvm_info("i2c_if", "ACK detected after initiator read data from target. Check device under test is not pulling SDA down!", UVM_HIGH)
        end
      end : read_byte_b

      begin : write_byte_b
        @(negedge scl);
        disable read_byte_b;

        `uvm_info("i2c_if", "Initiator writing data to target ...", UVM_HIGH)
        for(jj=6 ; jj>=0; jj=jj-1) // read 7 bits (MSB bit of the byte sampled above fork)
          begin
            @(posedge scl);
            reg_monitor = (reg_monitor | (sda << jj));  
          end
        data = reg_monitor;
        @(posedge scl); // Wait for ACK bit
        if (sda) begin
          `uvm_info("i2c_if", "NACK detected afer initiator wrote data to target.", UVM_HIGH)
          `uvm_info("i2c_if", "Write transaction unsuccessful.", UVM_MEDIUM)
          ack_return = ack;
        end
        else begin
          `uvm_info("i2c_if", "ACK detected after initiator wrote data to target.", UVM_HIGH)
        end
      end : write_byte_b
    join
    
	endtask 

  task sendDataToTarget(input logic [6:0] idToSend, 
                              logic [7:0] regAddrToSend, 
                              logic [7:0] dataToSend);

    // Start condition
    sda_drive = 1; 
    @(posedge scl);
    #sda_delay sda_drive = 0;
      
    // Write address
    for(ii=6; ii>=0; ii=ii-1) begin
      @(negedge scl);
      sda_drive = idToSend[ii];
    end
	  @(negedge scl);

    // Write rw_
    sda_drive = 0; // RW = 0
    @(posedge scl); // target reading rw_
    
    wait_for_ack();

    for(ii=7; ii>=0; ii=ii-1) begin
      @(negedge scl);
      sda_drive = regAddrToSend[ii]; 
    end

    wait_for_ack();

    for(ii=7; ii>=0; ii=ii-1) begin
      @(negedge scl);
      sda_drive = dataToSend[ii];
    end

    // Wait for ACK bit
    @(negedge scl);
    sda_drive = 1; // NACK should be detected by monitor
    @(negedge scl); // generate stop condition
    sda_drive = 0;
    @(posedge scl) #sda_delay sda_drive = 1;

  endtask

  bit [7:0] regData_out;
  task readDataFromTarget(input logic [6:0] idToSend, 
                                logic [7:0] regAddrToRead);
      
    // Start condition
    sda_drive = 1; 
    @(posedge scl);
    #sda_delay sda_drive = 0;
      
      // Write address
    for(ii=6; ii>=0; ii=ii-1) begin
	    @(negedge scl);
      sda_drive = idToSend[ii];
    end
    @(negedge scl);

    // Write rw_
    sda_drive = 0; // RW = 0
    @(posedge scl); // target reading rw_

    wait_for_ack();

    // Write register address
    for(ii=7; ii>=0; ii=ii-1) begin
	    @(negedge scl);
      sda_drive = regAddrToRead[ii];
    end
      
    wait_for_ack();
    @(posedge scl) sda_drive = 1; // repeated start
    #sda_delay sda_drive = 0;
	
    // Write address
    for(ii=6; ii>=0; ii=ii-1) begin
	    @(negedge scl);
      sda_drive = idToSend[ii];
    end
      
      
    sda_drive = 1; // RW = 1
    @(negedge scl) ; 
      
    wait_for_ack();
      
    // Read data from initiator
    for(ii=7; ii>=0; ii=ii-1) begin 
      @(posedge scl);
      regData_out = (regData_out | (sda << ii));  
    end
       
    @(negedge scl); 
    sda_drive = 1; // NACK sent to target
    @(posedge scl); // wait for target to receive the NACK

    @(negedge scl) sda_drive = 0; // stop condition      
    @(posedge scl) #sda_delay sda_drive = 1;
      
  endtask

  // Miscellaneous Tasks
  task read_byte();
    @(negedge scl); 
    reg_monitor= 8'b0;
    ack_return = nack;
    for(jj=7; jj>=0; jj=jj-1) begin 
      @(posedge scl);
      reg_monitor = (reg_monitor | (sda << jj)); 
    end

    @(posedge scl); // Wait for ACK bit
    if (sda) begin
      `uvm_info("i2c_if", "NACK detected from monitor task call", UVM_DEBUG)
      ack_return = nack;
      return;
    end
    else begin
      `uvm_info("i2c_if", "ACK detected from monitor task call", UVM_DEBUG)
      ack_return = ack;
    end
    `uvm_info("i2c_if", $sformatf("Read byte: %h", reg_monitor), UVM_DEBUG)
  endtask

  task wait_for_ack();
    @(negedge scl);
    sda_drive = 1; // release SDA
    @(posedge scl); 
    if (sda) begin // generate stop condition if NACK happens
      @(negedge scl) sda_drive = 0;
      @(posedge scl) #sda_delay sda_drive = 1;
      disable readDataFromTarget;
      disable sendDataToTarget;
    end 
  endtask

endinterface
