//  Analog model of differential clock driver
//  * Single ended input, differential output, 1.8V nominal swing 
//  * Amplitude adjustment (2-bits) and slew rate adjustment (2-bits)

`timescale 1ns/1fs
`ifdef XRUN
import cds_rnm_pkg::*;
`endif

`ifdef VCS
import snps_msv_nettype_pkg::*;
`endif


module clk_driver_diff (
  clkout_p,
  clkout_n,
  clkin,
  ampl_adj,
  sr_adj
  );
  
  output wreal4state clkout_p, clkout_n;
  input wreal4state clkin;
  input logic [1:0] ampl_adj, sr_adj;
  
  parameter real vsup = 1.8;        //Supply voltage for clock signal (V)
  parameter real ts = 0.1ps; // sampling rate during ramping    1ps, [unit: ns]

  parameter real ampl_scale = 0.25; //Amplitude scale factor, linear
                                    //Assuming vsup = 1.8, ampl_scale = 0.25
                                    //00 = 1.8*(1-0) = 1.8, 
                                    //01 = 1.8*(1-0.25) = 1.35,
                                    //10 = 1.8*(1-0.5) = 0.9,
                                    //11 = 1.8*(1-0.75) = 0.45
  parameter real slewp = 1.8e12; //Positive nom. slew rate (V/s) -> sr_adj = 0.   
  parameter real slewn = -1.8e12;//Negative nom. slew rate (V/s) -> sr_adj = 0
  parameter real sr_scale = 0.1;                //Slew rate scale factor, linear
                                                //Assuming slew = 1.8e12, sr_scale = 0.1
                                                //00 = 1.8e12*(1-0) = 1.8e12, 
                                                //01 = 1.8e12*(1-0.1) = 1.62e12,
                                                //10 = 1.8e12*(1-0.2) = 1.44e12,
                                                //11 = 1.8e12*(1-0.3) = 1.26e12
  parameter real slew_step = 5; // ramping step of the output of slew rate limiting block
	
  reg clk_int;                      //Variable for internal clock
  real clkout_vmax;                 //Maximum output voltage, based on ampl_adj value
  real slewp_scaled, slewn_scaled;  //Scaled slew rates
  real tr, tf; 											// rise time and fall time of the ramp
  int nstep; 												// number of full steps in ramp
  real td; 												  // fractional step to finish ramp
  real vinc;												// coefs for ramping  V
  
  always begin
    case(ampl_adj)
      0:        clkout_vmax = vsup;
      1:        clkout_vmax = vsup * (1.0 - ampl_scale);
      2:        clkout_vmax = vsup * (1.0 - 2.0 * ampl_scale);
      3:        clkout_vmax = vsup * (1.0 - 3.0 * ampl_scale);
      default:  clkout_vmax = 0;
    endcase
    @(ampl_adj);
  end
  
  always begin
    case(sr_adj)
      0:  begin
            slewp_scaled = slewp;
            slewn_scaled = slewn;
          end
      1:  begin
            slewp_scaled = slewp * (1.0 - sr_scale);
            slewn_scaled = slewn * (1.0 - sr_scale);
          end
      2:  begin
            slewp_scaled = slewp * (1.0 - 2.0 * sr_scale);
            slewn_scaled = slewn * (1.0 - 2.0 * sr_scale);
          end
      3:  begin
            slewp_scaled = slewp * (1.0 - 3.0 * sr_scale);
            slewn_scaled = slewn * (1.0 - 3.0 * sr_scale);
          end
      default:  begin
                  slewp_scaled = slewp * 1e-9;
                  slewn_scaled = slewn * 1e-9;
                end
    endcase
    @(sr_adj);
  end
  
  real clkin_scaled, clkout_p_int, clkout_n_int; //Variables for scaled input clock and output clocks
  real clkin_diff_norm;
  assign clkin_diff_norm = (clkin - (vsup/2.0)) * (2.0/vsup); // goes from -1 to 1
  assign clkin_scaled = (clkin_diff_norm * clkout_vmax);


  slew #(.slew_steps(slew_step)) srlp(.in(clkin_scaled), .out(clkout_p_int), .slewp(slewp_scaled), .slewn(slewn_scaled));
  
  assign clkout_p = clkout_p_int;
  assign clkout_n = -clkout_p_int;
  

endmodule

